#!/usr/bin/perl
###############################################################################
# intensitytable.pl:  Produces the graphical intensity probability table
#
# Usage:  cronjob, perl intensitytable.pl
#
# Input:  PWS text probability message from 2008 or later
#
# Output: Intensity table graphic for display on the web
#
# Required Software: perl, imagemagick convert
#
# Methodology:
#       This script scans the pws directory from the /mnt/atcf file system
#       for wind speed probability text products modified recently (within
#       the last 3 hours).  If a text product is found and no graphic has
#       been produced for this storm and advisory number (graphic does not
#       exist in the archive subdirectory) the script parses the text
#       product for the intensity table and translates it into graphical
#       format using imagemagick convert to populate the template graphic
#       with text values.
#
#
# Author:           Chris Lauer, National Hurricane Center, Miami, FL
# Maintained by:    Chris Lauer
# Revision History:
#                   20080514:  Implemented (Chris Lauer)
#                   20080616:  Modified to work in HPC environment (Klein)

# Robson   09/30/11   RHEL5 tested.
# Robson   07/12/13   Modified for WCOSS. Now pulls files from NCOSRV. CF cron job moves
#                     files from WCOSS to NCOSRV.

$date = time;
use Time::Local;   #Needed for date/time manipulation
use File::Path;    #Needed for working directory creation
use File::Copy;

$home = $ENV{'HOME'};
$NCOSRV = $ENV{'NCOSRV'};
$NETAPPIP = $ENV{'NETAPPIP'};
$programDir = "/export-4/$NETAPPIP/hpcops/grids/atcf/intensitytable";
$archiveDir = "$programDir/archive";
$launchDir = "/export-5/$NETAPPIP/hpcops/ops/scripts/interactive";
$tabledir = "/export-5/$NETAPPIP/hpcops/tables/tpc/stns";

$ENV{'DISPLAY'} = ":0.0";
$ADVS = "$home/wgraph";
$pwsDir = "$ADVS/wndprb";
$workDir = "$ADVS/work_intensitytable";

# Make sure that there is a wgraph directory off of the home directory.
# And make the working directory.
#
mkpath($pwsDir,0,0777);
mkpath($workDir,0,0777);
chdir($ADVS);

# Pull files from NCOSRV for this graphic. Only need wndprb directory!

$proddir = "/pub/share/hpcops/grids/atcf"
system ("pullsync hpcops $NCOSRV ${proddir}/wndprb ~/wgraph/wndprb/. D");

# Check to make sure there is a wndprb directory
#
if(opendir(PWSDIR,$pwsDir)){
	
}else{
	print("Could not open folder containing ATCF probability advisories: '$pwsDir'\n");
	print("aborting script..\n");
	
	exit;
}

# Read all of the files from the wndprb directory.
#
@files = readdir(PWSDIR);

close(PWSDIR);

# Loop through the files...looking for the most recent one.
foreach $file (@files){
	$fileAbsPath = "$pwsDir/$file";
	$textProductFileName = $fileAbsPath;
 	if(($file =~ /^(al|ep)(\d\d)(\d\d)(\d\d)\.wndprb\.(\d\d\d)$/) && (-M $fileAbsPath < .25)){
		$basin          = $1;
		$BASIN          = uc($basin);
		$stormNumber    = $2;
		$yyyy           = $3.$4;
		$yy 		= $4;
		$advisoryNumber = $5;
		if($stormNumber >=50){
			next;
		}
		if($BASIN eq "AL"){
			$webDirName = "AT$stormNumber";
		}else{
			$webDirName = "EP$stormNumber";
		}
                print $filePrefix;
		$filePrefix = $BASIN . $stormNumber . $yyyy;
		$webArchiveFileName = $filePrefix . "T." . $advisoryNumber . ".GIF";
		$webUpdateFileName = $BASIN . $stormNumber . $yy . "T.GIF";
		if(-s "$archiveDir/$webArchiveFileName"){
			print("Already processed graphics for $file\n");
			next;
		}else{
                        print "going to function";
			&runGraphic;
		}
	}
}
sub runGraphic{
        mkpath($workDir,0,0777);
        chdir($workDir);
	&parseTextProduct;
	&computeTime;
	&drawTable;
	#send to web
	print("Sending:  $webArchiveFileName to $webDirName/$webUpdateFileName \n");
 	system("perl ${launchDir}/web_launch_hpc.pl $webArchiveFileName graphics $webDirName/$webUpdateFileName");
        print("\n");
 	system("perl ${launchDir}/web_launch_hpc.pl $webArchiveFileName graphics /home/httpadm/docs/archive/$yyyy/graphics/$basin$stormNumber/$webArchiveFileName");
 	system("mv $webArchiveFileName $archiveDir");
	unlink("last.nts");
	unlink("gemglb.nts");
	rmdir($workDir);
}
sub drawTable{
	$numRows = 11;
	$numColumns = 8;
	@columnOffset = ("+12", "-155", "-63", "+28", "+122", "+215", "+308", "+399");
        #---------------------------------------------------------------
        # As has become standard, we (HPC) don't have the latest version
        # of imagemagick, so "annotate" doesn't work, but "draw" does
        #---------------------------------------------------------------
	$text = "convert $programDir/static_gif/wind_table_template.gif -font Helvetica -pointsize 22 -fill black";
 	$text .= " -gravity north -annotate 0x0+0+30 'Intensity (Maximum Wind Speed) Probability Table'";
        $text .= " -annotate 0x0+0+60 '$stormName Advisory Number $advisoryNumber'";
        $text .= " -annotate 0x0+0+90 '$advisoryTimeLine'";
        $text .= " -pointsize 22";

	$rowOffset = 213;
	$pointsize = "-pointsize 19";
	@row = @fHours;
	&plotRow;
	$rowOffset = 235;
	$pointsize = "-pointsize 14";
	@row = ("","for","for","for","for","for","for","for");
	&plotRow;

	$rowOffset = 250;
	$pointsize = "-pointsize 17";
	@row = @fTimes;
	&plotRow;

	$pointsize = "-pointsize 22";
	$rowOffset = 290;
	@row = @Dissipated;
	&plotRow;

	#$rowOffset = 325;
	$rowOffset += 42;
	@row = @Depression;
	&plotRow;

	$rowOffset += 42;
	@row = @Storm;
	&plotRow;

	$rowOffset += 42;
	@row = @Hurricane;
	&plotRow;
	$lineColumnOffset = 244;

	$rowOffset += 42;
	@row = @Hurricane1;
	&plotRow;

	$rowOffset += 42;
	@row = @Hurricane2;
	&plotRow;

	$rowOffset += 42;
	@row = @Hurricane3;
	&plotRow;

	$rowOffset += 42;
	@row = @Hurricane4;
	&plotRow;

	$rowOffset += 42;
	@row = @Hurricane5;
	&plotRow;

	$rowOffset = 676;
	@row = @maxWindMph;
	&plotRow;

 	system("$text $webArchiveFileName");

}
sub parseTextProduct{
	if(open(WNDPRB, $textProductFileName)){
		@lines = <WNDPRB>;
		close(WNDPRB);
	}else{
		print("Could not open text product:  $textProductFileName\n");
	}
	foreach $line (@lines){
		if($line =~ /^(.*) WIND SPEED PROBABILITIES NUMBER\s*(\d+)\s*$/){
			$stormName = $1;
			$advisoryNumber = $2;
			$stormName = &mixCase($stormName);
		}elsif($line =~ /^(\d\d)(\d\d) UTC \w\w\w (\w\w\w) (\d\d) (\d\d\d\d)\s*$/){
			$hhs  = $1;
			$mm   = $2;
			$mon  = $3;
			$dd   = $4;
			$yyyy = $5;			
		}elsif($line =~ /...LONGITUDE (\d?\d\d\.\d) (WEST|EAST) /){
			$centerLon = $1;
			$lonSign = $2;
			if($lonSign eq "WEST"){
				$centerLon *= -1;
			}
		}elsif($line =~ /^DISSIPATED\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Dissipated = ("Dissipated", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^TROP DEPRESSION\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Depression = ("Tropical Depression (<39)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^TROPICAL STORM\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Storm = ("Tropical Storm (39-73)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^HURRICANE\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Hurricane = ("Hurricane (all categories)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^HUR CAT 1\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Hurricane1 = ("    -- Category 1 (74-95)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^HUR CAT 2\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Hurricane2 = ("    -- Category 2 (96-110)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^HUR CAT 3\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Hurricane3 = ("    -- Category 3 (111-130)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^HUR CAT 4\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Hurricane4 = ("    -- Category 4 (131-155)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^HUR CAT 5\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*(X|\d{1,2})\s*$/){
			@Hurricane5 = ("    -- Category 5 (>155)", "$1%", "$2%", "$3%", "$4%", "$5%", "$6%", "$7%");
		}elsif($line =~ /^FCST MAX WIND\s*(\d{1,3})KT\s*(\d{1,3})KT\s*(\d{1,3})KT\s*(\d{1,3})KT\s*(\d{1,3})KT\s*(\d{1,3})KT\s*(\d{1,3})KT\s*$/){
			@maxWindKts = ($1,$2,$3,$4,$5,$6,$7);
			push(@maxWindMph,"Forecast Maximum Wind");
			foreach $maxWindKt (@maxWindKts){
				$fstMaxWindMPH = int(($maxWindKt*1.1516 + 2.5)/5)*5;

				if($fstMaxWindMPH == 130){ 
					$fstMaxWindMPH = 135; 
				}
                                if($fstMaxWindMPH == 0){
				   push(@maxWindMph,"--");
				}else{
				  	push(@maxWindMph,"$fstMaxWindMPH mph");		
				}
			}
			
		}elsif($line =~ /^  TIME       (\d\d)Z \w\w\w/){
			$hh = $1;
		}	
	}	
}
sub computeTime{
	@months = ("Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec");  	# array of months for int to month string conversion
	@weekDays = ("Sun","Mon","Tue","Wed","Thu","Fri","Sat");
	for($i=0;$i<12;$i++){
		if($mon eq uc($months[$i])){
			$cMonth = $i;			
		}
	}	
	#Detect Standard vs Daylight Time
	$ENV{"TZ"} = "EST5EDT";		#set time zone variable to eastern daylight/standard
	$dateForZone = `date`;		#use system date to get date/time
	if($dateForZone =~ /EST/){
		$dst = "0";		
	}
	elsif($dateForZone =~ /EDT/){
		$dst = "1";
	}
		if($basin eq "ep"){
		if($dst){
			$zone = "PDT";
			$zoneOffset = -7;
		}else{
			$zone = "PST";
			$zoneOffset = -8;
		}
	}elsif($centerLon <= -85){
		if($dst){
			$zone = "CDT";
			$zoneOffset = -5;
		}else{
			$zone = "CST";
			$zoneOffset = -6;
		}
	}elsif($centerLon <= -70){
		if($dst){
			$zone = "EDT";
			$zoneOffset = -4;
		}else{
			$zone = "EST";
			$zoneOffset = -5;
		}
	}else{
		$zone = "AST";
		$zoneOffset = -4;
	}
	
	$year = $yyyy - 1900;	
	$stormTimeS = timegm(0,$mm,$hhs,$dd,$cMonth,$year);	# get time (in seconds from epoch) in zulu for the advisory synoptic time
	$stormTimeS = $stormTimeS + (3600*$zoneOffset);			# adjust time to storm's time zone
	($cursec,$curmin,$curhr,$curday,$curmon,$curyear,$curwd,$curyd,$isdst) = gmtime($stormTimeS);  #compute time variables for forecast time
		# Adjust 0-23 hour to 12 hours with AM/PM
		if($curhr >= 12){
			$AMorPM = "PM";
			if($curhr > 12){
				$curhr -= 12;
			}
		}else{
			$AMorPM = "AM";
		}		
		if($curhr == 0){
			$curhr = 12;
		}
	$yyyyS = $curyear + 1900;
	$advisoryTimeLine = "$curhr:$mm $AMorPM $zone $months[$curmon] $curday $yyyyS";
	$stormTime = timegm(0,0,$hh,$dd,$cMonth,$year);	# get time (in seconds from epoch) in zulu for the advisory synoptic time
	$stormTime = $stormTime + (3600*$zoneOffset);			# adjust time to storm's time zone
	push(@fHours,"");
	push(@fTimes,"");
	foreach $fHour (12,24,36,48,72,96,120){
		push(@fHours,"$fHour hour");
		$fTime = $stormTime + 3600*$fHour;
		($cursec,$curmin,$curhr,$curday,$curmon,$curyear,$curwd,$curyd,$isdst) = gmtime($fTime);  #compute time variables for forecast time
		# Adjust 0-23 hour to 12 hours with AM/PM
		if($curhr >= 12){
			$AMorPM = "PM";
			if($curhr > 12){
				$curhr -= 12;
			}
		}else{
			$AMorPM = "AM";
		}		
		if($curhr == 0){
			$curhr = 12;
		}
		push(@fTimes,"$curhr $AMorPM $weekDays[$curwd]");						
	}
}
sub plotRow{
	$text .= " $pointsize -gravity north";
	for($i = 1; $i<$numColumns; $i++){
		if($row[$i] eq "X%"){ $row[$i] = "<1%"; } 
 	        $text .= " $gravity -annotate 0x0$columnOffset[$i]+$rowOffset '$row[$i]'";		
	}
}
sub mixCase{
	my $origString = @_[0];
	my @origStrings = split(/ +/,$origString);
	my $returnString = "";
	foreach $origString (@origStrings){		
		if($origString =~ /^(\w)(.*)/){
			$origString = uc($1) . lc($2);
		}
		if($origString =~ /(.*)-(\w)(.*)/){
			$origString = $1 . "-" . uc($2) . lc($3);
		}
		$returnString .= $origString . " ";
	}
	chop($returnString);
	return $returnString;
}




