#!/usr/bin/perl

###################################################################################
#  Program:  web_launch_hpc.pl
#
#  web_launch.pl sends TAFB and NHC products to all required NHC/TPC
#   webservers via scp, using public/private key authentication.  The
#   webservers to send products too are stored in the @webservers
#   array.
#
#  web_launch.pl takes 3 arguments:
#
#  1)  $fileName:  The local filename of the graphic to launch 
#                   (e.g. danger_atl.gif)
#                   Multiple file names can be provided in single quotes or | (pipe) 
#                   delimited (e.g. danger_atl.gif|danger_pac.gif).
#  2)  $account:   The account to login to the webserver as
#                   (e.g. tafb, graphics, etc.)
#  3)  $directory: The directory on the remote machine to place the images in
#                   (e.g. /home/graphics/AT03, ~/AT03)
#
#      Author:  Chris Lauer
#        
#      Revision History:
#
#         7-24-06:  Implemented to send graphics to largo.nhc.noaa.gov
#             -C.Lauer
#
#        10-02-06: Added logging; use of | character for multiple files at once;
#                   color coded status messages and blinking error message
#             -C.Lauer
#  07/18/07   A.Robson/HPC    Modified for HPC
#  08/12/10   A.Robson/HPC    Added name of new backup webserver: vp-tpcbkup.ncep.noaa.gov
#  01/11/11   A.Robson/HPC    Changed to new webserver.
#  09/30/2011 A.Robson/HPC    RHEL5 testing.
#  05/17/2012 A.Robson/HPC    Change mail server.
#  06/12/2012 A.Robson/HPC    Added new environment variable for NHCBKUPWEB server

###################################################################################

$NHCBKUPWEB=$ENV{'NHCBKUPWEB'};  # The server address is now set in the .chsrc
# NCWCP: vm-lnx-nhcbkup.ncep.noaa.gov, 
# WWB vp-tpcbkup.ncep.noaa.gov




use Term::ANSIColor; # pretty colored terminal
$bindir = "/accounts/graphics/bin";  # set directory for bins (vsmail, ncftpput, and logging)

$numargs = @ARGV;  # get number of arguments.
if($numargs != 3){ # if number of arguments is not 3, print usage information and exit
	print("\n\nUSAGE:  web_launch.pl fileToSend username targetdirectory\n
		\n");
	&badlaunch;
	exit;
}else{                          #otherwise, assign arguments to variables
	$fileName  = $ARGV[0];  #first argument is the file to send
	$fileName =~ s/\|/ /g;  #allow multiple files to be sent in one instance, separated by a pipe (replace pipe with space)
	$account   = $ARGV[1];  #second argument is the account to send to
	$directory = $ARGV[2]; 	#third argument is the remote directory to send to, '~/' for home
}

# List of webservers to launch to (e.g. seahorse.nhc.noaa.gov, 140.90.176.xx, etc.)
#@webservers = ("largo.nhc.noaa.gov", "seahorse.nhc.noaa.gov", "ratfish.nhc.noaa.gov");  
#@webservers = ("lnx131.ncep.noaa.gov"); # OLD HPC BACKUP WEBSERVER
#@webservers = ("vp-tpcbkup.ncep.noaa.gov"); # CURRENT HPC BACKUP WEBSERVER FOR NHC
@webservers = ("$NHCBKUPWEB"); # NEW NHC HPC BACKUP WEBSERVER FOR NHC

# Set up date/time for logging purposes
($cursec,$curmin,$curhr,$curday,$curmon,$curyear,$curwd,$curyd,$isdst) = gmtime();
$yyyy = $curyear + 1900;
$mm = $curmon + 1;
$mm = sprintf("%02d",$mm);
$dd = sprintf("%02d",$curday);
$hhmmss = sprintf("%02d:%02d:%02d",$curhr,$curmin,$cursec);

# Set up log file name
$logFileName = "$bindir/logs/$yyyy$mm$dd\_web_launch.log";
$LOG = "# $hhmmss ##################################################################\n";
$pwd = $ENV{'PWD'};
$host = $ENV{'HOSTNAME'};
$user = $ENV{'USER'};
$LOG .= "$fileName to $account:$directory from $user at $host in $pwd\n";

# Set up mail information
$server = "smtp.gmail.com";
$sender = "hpcWeblaunch\@noaa.gov";
@recipients = ("Alan.Robson\@noaa.gov");
$subject = "Product Web Launch Failure $hhmmss: $fileName to $account:$directory from $user at $host in $pwd.";
$sendMessage = 0;

# Loop through each webserver and launch product
foreach $webserver (@webservers){
	print("Sending to web ($webserver)...");
	# Use scp in batch mode (-B) to prevent hang on auth errors, etc. - also disable
	# strict host key checking (always checking against ~/.ssh/known_hosts) to prevent
	# failure on a bad known_hosts file (graphics are meant for the public, so this is
	# not a security risk).  Use 2>&1 to redirect standard error into $ssh as well. -v
	# option is also turned on to allow diagnosis of whether transfer was successful.	
	$ssh = `/usr/bin/scp -B -v -o StrictHostKeyChecking=no $fileName $account\@$webserver:$directory 2>&1`;
	if($ssh =~ /Exit status 0/){ # Exit status 0 means transfer was successful
		print color 'green bold'; # print green
		print " success.\n"; # success message
	}else{					# Otherwise, transfer failed
		print color 'red blink bold';	# print blinking red
		print " FAILED!!!!!!!!!\n";     # failure message
		$LOG .= "$webserver FAILED!!!!  Verbose log follows:\n$ssh";
		$LOG .= "\n/////////////////////////////////////////////////////////////\n";
		$sendMessage = 1;
	}

	print color 'reset';			# reset colors to normal
}

if(open(LOG,">>$logFileName")){
	$LOG .= "_________________________________________________________________________\n";
	print LOG ("$LOG\n");
	close(LOG);	
}else{
	print color 'red';
	print "Error: Could not open log file: $logFileName ";
	print color 'reset';
	print "(non-critical error, alert TSB if present or make log entry)\n";
}

if($sendMessage){
	foreach $recipient (@recipients){
		$vsmail = `$bindir/vsmail $server $sender $recipient "$subject" "$LOG" 2>&1`;
	}
}

sub badlaunch {
	($cursec,$curmin,$curhr,$curday,$curmon,$curyear,$curwd,$curyd,$isdst) = gmtime();
	$yyyy = $curyear + 1900;
	$mm = $curmon + 1;
	$mm = sprintf("%02d",$mm);
	$dd = sprintf("%02d",$curday);
	$hhmmss = sprintf("%02d:%02d:%02d",$curhr,$curmin,$cursec);

	# Set up log file name
	$pwd = $ENV{'PWD'};
	$host = $ENV{'HOSTNAME'};
	$user = $ENV{'USER'};
	$server = "outbound.nems.noaa.gov";
	$sender = "hpcWebLaunch\@noaa.gov";
	$recipient = "Alan.Robson\@noaa.gov";
	
	$message = "Bad launch:  $user at $host in $pwd ($hhmmss)";
	$vsmail = `$bindir/vsmail $server $sender $recipient "$message" "$message" 2>&1`;	

}
