#!/usr/bin/perl

################################################################################
# program windprob.pl
#	usage:  perl windprob.pl (call from cron every minute)
#
#	windprob.pl checks for wind speed probability grib files from either:
#
#	1) IBM (final version, all storms)
#	2) ATCF (preliminary version, single storm)
#
#	If data exists, converts to GEMPAK and uses getStormInfo.pl to generate 
#	wind speed probability graphics, archives grid, and makes data
#	available to operations (scp to shad).
#
#	Program uses a lock file ($lockFile) to avoid being run concurrently
#
#	Author:  Chris Lauer
#		 National Hurricane Center
#		 Miami, FL
#
#	History:  3-30-2007:  Completed, documented.
#                 7-17-2008:  Modified for use at HPC for 2008 season. (Klein)
#                             View the README file $hpc_opsdir below for additional details.
################################################################################

use File::Path;    #Needed for directory creation

my $homeDir = $ENV{HOME};
my $OPSDIR  = $ENV{OPSDIR};
my $WINDPROB= $ENV{WINDPROB};
$hpc_opsdir = "$OPSDIR/wndprb_2008"; # Directory containing all scripts for wind probs.
$incomingdir = "$WINDPROB";         # Directory on ncosrv containing data for the wind probs.
$scriptsdir = "$homeDir/wgraph/wndprb/scripts"; # Directory where scripts will be copied to on local box.
mkpath($scriptsdir,0,0777);                     # Make this scripts directory
system ("cp -Rp $hpc_opsdir/* $scriptsdir");    # Copy the scripts from ncosrv.
require "$scriptsdir/windProbs_config.pl";      # configuration file for all perl scripts.

$binDir = $Config::binDir;; # Directory where scripts and programs are found

$prelimIncomingDir = $incomingdir;  # Directory where incoming preliminary windprob grids and adv.txt files are placed
                                    # Note:  HPC does not receive these prelim windprob grids.
$IBMGribIncomingDir = $incomingdir;  # Directory where incoming final windprob grids and DTG.corners files are placed

$prelimArchiveDir = $Config::gribDir;; # Directory where preliminary grids are moved to (archived)
$IBMGribArchiveDir = $Config::gribDir;; # Directory where final grids are moved to (archived)

$gemGridDir = $Config::gridDir; # Directory where converted (from grib) gempak grids are placed

# Below are settings for additional directories that we'll need to pull data over 
# from the tpcprd account on the CCS.

$aiddir = $Config::aidDir;;
$advdir = $Config::advDir;;
$workingdir = $Config::workDir;;
mkpath($gemGridDir,0,0777);
mkpath($aiddir,0,0777);
mkpath($advdir,0,0777);

# Set locking file name/location.  This file will be unlinked when the program is done.
$lockdir = $Config::probDir;;
$lockFile = "$lockdir/windprob.lock";   # File name of lock file to use (don't want overlapping processing)

&checkLock; # Check for existence of lock/create lock file

$ENV{'DISPLAY'} = ":1.0";  # Set display for GEMPAK

# Remove old working directory and create new one
system("rm -Rf $workingdir");
mkpath($workingdir,0,0777);
chdir($workingdir);

# Get input from "fourpanel".  This will be the date/time and cycle of the 
# file to process.

$datetime=$ARGV[0];


# Process new grib files sent from the IBM.  These are in $incomingdir set above.
# For HPC, it is possible that the grib data will not be ready at the time this script is 
# called.  As a result, need to sleep for a period of time.  If not in after an hour, force
# script to exit.

$found="no";
$lpcount=0;
while ($found eq "no" && $lpcount < 13 ) {

    if(checkIBM()){
	print("Finished processing IBM Grib\n");
        $found="yes";
    }else{
	print("No new grib from IBM found...sleeping for 5 minutes\n");
        system("sleep 300");	
    }
    $lpcount++;
}

#Process new prelim grids sent from the ATCF.  As of 2008 season, HPC does not get these prelim grids.
if(checkPrelim()){	
	print("Finished processing Prelim Grib\n");
}else{
	print("No new preliminary gribs from ATCF found\n");
}

#Processing is complete - release lock file.
&releaseLock;

################################################################################
# Subroutine - checkLock 
# Checks for the existence of lock file ($lockFile).  If exists, exits program
# If no lock file exists, creates lock file ($lockFile).  If for some reason
# lock file does not exist and can not be created, processing is aborted (panic)
################################################################################
sub checkLock{
	if(-s $lockFile){
		print("Lock file found, aborting processing..\n");
		exit;
	}else{
		if(open(LOCK,">$lockFile")){
			print LOCK ("Lock file for wind speed probability processing");
			close(LOCK);
		}else{
			print("Could not open Lock file!  Aborting processing (abnormal condition)\n");
			exit;
		}
		
	}		
}


################################################################################
# Subroutine - releaseLock :
# Delete lock file and exits program
################################################################################
sub releaseLock{
	unlink($lockFile);
	exit;
}

################################################################################
# Subroutine - checkIBM :
# Checks for the existence of a final wind speed probability (produced by IBM).
# If the file exists, convert to GEMPAK format, move to archive folder, run 
# getStormInfo to compute plot information and then run the driver program to
# plot the data.  Return 1 (True) if a grib exists and is processed.
################################################################################
sub checkIBM{
	my $return = 0; #initialize return to false
	if(opendir(DIR, "$IBMGribIncomingDir")){  #Open directory
		@files = sort readdir(DIR);  #Read list of files from directory
	}else{
		print("Could not open Incoming IBM Grib data directory: $IBMGribIncomingDir\nExiting...");  #If the directory couldn't be opened, warn
		&releaseLock;		#exit program

	}
	closedir(DIR);
        $DTG=$datetime;
	
	#Loop through the files in the final wind speed probability directory
	foreach $file (@files){
		#if($file =~ /^tpcprblty\.(\d{10})\.grib1$/){  #If file matches this pattern, process
		if($file =~ /^tpcprblty\.$DTG\.grib1$/){  #If file matches this pattern, process
			#$DTG = $1;	#retain Date-Time Group
			print("IBM Grib found for: $DTG.. processing\n"); 
			$GBFILE = "$IBMGribIncomingDir/$file"; #set GBFILE to grib file name (for GEMPAK)
			$GDOUTF = "$gemGridDir/tpcprb_$DTG"; #set GDOUTF to output gempak grid file name
                        &convertGrib; #Convert the grib1 file to gempak format
		
			system("cp $IBMGribIncomingDir/$DTG.corners $gemGridDir/$DTG.corners"); #keep the corners file with the gempak file - will need it
			system("cp $IBMGribIncomingDir/$file $IBMGribArchiveDir"); # archive the ibm grib1 (move the file out of the incoming directory so it is not re-processed  
#			system("scp $GDOUTF naprod\@shad:/model/tpcprbs");
#			system("scp $GDOUTF naprod\@compute1:/model/tpcprbs");		

			system("/usr/bin/perl $binDir/getStormInfo.pl tpcprb_$DTG"); #Generate plot information file and plot
    
			$return = 1; #return 1 (true), a grib was processed
					
		}
	}
	
	
	
	return $return; 
}

################################################################################
# Subroutine - checkPrelim :
# Checks for the existence of a prelim wind speed probability (produced by ATCF).
# If the file exists, convert to GEMPAK format, move to archive folder, run 
# getStormInfo to compute plot information and then run the driver program to
# plot the data.  Return 1 (True) if a grib exists and is processed.
################################################################################
sub checkPrelim{
	my $return = 0; #initialize return to false
	if(opendir(DIR, $prelimIncomingDir)){  #Open directory
		@files = sort readdir(DIR);  #Read list of files from directory
	}else{
		print("Could not open Incoming IBM Grib data directory: $prelimIncomingDir\nExiting...");  #If the directory couldn't be opened, warn
		&releaseLock; #exit program
	}

	closedir(DIR);  

	#Loop through the files in the prelim wind speed probability directory
	foreach $file (@files){
		if($file =~ /^(\w\w)(\d\d)(\d{4})_(\d{10})\.grib$/){ #If file matches this pattern (stormid_datetimegroup.grib), process
			$stormNumber = $1 . $2 . $3; #retain storm number (i.e. al032006)
			$DTG = $4; #retain Date-Time Group
			print("Prelim Grib found for storm $stormNumber at $DTG.. processing\n");
			$gemFileName = "$stormNumber" . "_$DTG";
			$GBFILE = "$prelimIncomingDir/$file"; #set GBFILE to grib file name (for GEMPAK)
			$GDOUTF = "$gemGridDir/$gemFileName"; #set GDOUTF to output gempak grid file name
			if(-s "$prelimIncomingDir/$stormNumber.adv.txt"){
				&convertGrib;	#Convert the grib1 file to gempak format		


				system("cp $prelimIncomingDir/$file $prelimArchiveDir"); # archive the atcf grib1 (move the file out of the incoming directory so it is not re-processed
				system("cp $prelimIncomingDir/$stormNumber.adv.txt $gemGridDir");
				system("/usr/bin/perl $binDir/getStormInfo.pl $gemFileName"); #Generate plot information file and plot
				$return = 1; #return 1 (true), a grib was processed
			}else{
				print("No adv.txt for $stormNumber, not processing\n");
			}
		}
	}
	return $return;
}

################################################################################
# Subroutine - convertGrib :
# Invokes nagrib_nc to convert a grib1 file to gempak format
# $GBFILE is the grib file name (full path)
# $GDOUTF is the gempak file name (full path)
################################################################################
sub convertGrib{
	unlink("$GDOUTF"); # delete gempak grid file (in event of a re-run or special, need fresh data)
 
        # Copy over the ncepgrib2.tbl table to properly identify the grib data.
        # Locating this file in the scripts directory for simplicity's sake.
	
        system ("cp $binDir/ncepgrib2.tbl .");

	#Run nagrib_nc to convert, supress output.
	system("csh -c 'nagrib_nc << EOF
	 GBFILE   = $GBFILE
	 INDXFL   =
	 GDOUTF   = $GDOUTF
	 PROJ     = mer
	 GRDAREA  = dset
	 KXKY     = 10;10
	 MAXGRD   = 200
	 CPYFIL   = gds
	 GAREA    = dset
	 OUTPUT   = T
	 GBTBLS   = ncepgrib2.tbl
	 GBDIAG   =
	 PDSEXT   = NO 
	 r\n 
	 e\n
         EOF\n\n'");
}
