#!/usr/bin/perl

################################################################################
# program getStormInfo.pl
#	usage:  perl getStormInfo.pl yyyymmddhh
#
#	getStormInfo.pl gathers all the information about storms active at the
#	command line specified date/time group and prints the information to
#	a text file.  The information is intended to be used as input along with
#	the wind speed probalities grid file to generate the wind speed
#	probability graphics for all active storms.
#
#	Author:  Chris Lauer
#		 National Hurricane Center
#		 Miami, FL
#
#	History:  4-27-2006:  Completed, documented.
#                 8-07-2006:  Fixed bug where corners file generated on the ibm
#                             was not processed correctly and resulted in a 
#                             white graphic (bad plot boundaries).  Also added
#                             code to adjust plot boundaries to include the
#                             storm center position, if necessary.
#                 7-21-2008:  Configure to run at HPC for 2008 season. (Klein)
#                 7-15-2013:  Modified for WCOSS (Robson)
#
################################################################################
use Math::Trig;    #Needed to convert lat/lon to mercator coordinates (aspect ratio calculation)
use Time::Local;   #Needed for date/time manipulation

my $homeDir = $ENV{HOME};
my $NCOSRV  = $ENV{NCOSRV};
$scriptsdir = "$homeDir/wgraph/wndprb/scripts"; # Directory on local box where all perl scripts are located.
require "$scriptsdir/windProbs_config.pl";	# configuration file

## Set up directories
$probDir = $Config::probDir;			# top level directory for probabilities
$dataDir = $Config::gribDir;			# directory containing corners files
$gridDir = $Config::gridDir;			# directory containing grid files
$workDir = $Config::workDir;			# work directory (end plotParams file saved here)
$binDir  = $Config::binDir;			# bin/scripts directory

$atcfDir = $Config::atcfDir;			# directory containing ATCF advisories/adecks
$marDir = $Config::marDir;			# marine advisory directory
$pubDir = $Config::pubDir;			# public advisory directory
$disDir = $Config::disDir;			# discussion directory
$aidDir = $Config::aidDir;			# a-deck directory
$advDir = $Config::advDir;			# advisory text file directory

$plotParams = $Config::plotParams;		# plotParams file (this program's output)

unlink($plotParams);				# delete plotParams file (to avoid using previous plotParams file)


# Determine which machine is production and pull over any files in the aid directory.
# Currently, the adv directory is not created on the CCS...so this is commented out. 

#$PRODCCS = (`ping -c 1 prodccs | head -1 | awk '{print \$2}' | awk -F"." '{print \$1}'`);
#chomp($PRODCCS);
$proddir = "/tpc/noscrub/data/atcf-noaa";

system ("pullsync hpcops ${NCOSRV} ${proddir}/aid ~/wgraph/aid/. D");
#system ("pullsync hpcops ${PRODCCS} ${proddir}/aid ~/wgraph/aid/. D");

#TESTING BLOCK - sets some variables to the wrong thing
#$marDir = "/home/devel/incoming/prelim/atcf/mar";
#$pubDir = "/home/devel/incoming/prelim/atcf/pub";
#$advDir = "/home/devel/incoming/prelim";

#Set up date/time information
$gridFileName = $ARGV[0];			 	 # Get file name from command line
if ($gridFileName =~ /^(.*)_(\d\d\d\d)(\d\d)(\d\d)(\d\d)$/){	 # Parse for DTG and grid type
	$fcstAdvDTG = "$4/$5" . "00Z";		 # to reformat into the same format as the forecast advisory
	$DTG = $2 . $3 . $4 . $5;
	$gridType = $1;				#gridType is tpcprb for ibm grids, stormnumber for individual storms
	
	if($gridType eq "tpcprb"){
		$cornerFile = "$gridDir/$DTG.corners";		# File name for the corners file (describes the contents of the wind speed probability grids)
		if(-s $cornerFile){				# If the corners file exists
			&processCornersFile;			# get storm numbers and probability boundaries from the corners file
		}else{						# if corners file does not exist..
			print("No corners file - taking next best guess\n");	
			&processNoCorners;			# find active storm numbers and guess at plot boundaries through an alternative method
		}
		$singleStorm = 0;
	}else{
		$advDir = $gridDir;
		&computeCorners;
		push(@stormIDs,$gridType);
		$numStorms = 1;
		$singleStorm = 1;
	}
}else{						 # if no DTG is specified on the command line, print usage info
	   $usage  = "\n";   
	   $usage .= "---------------------------------------------------------\n";
	   $usage .= "Usage:  perl getStormInfo.pl grib1filename\n";
	   $usage .= "        where grib1filename is of the form:\n";
	   $usage .= "           tpcprb_YYYYMMDDHH or bb##YYYY_YYYYMMDDHH\n";
	   $usage .= "        of the wind speed probability grid\n";
	   $usage .= "        e.g.  perl getStormInfo.pl al012005_2005061818\n";
	   $usage .= "---------------------------------------------------------\n";
	   
	   die($usage); # exit, print usage.
}



# Loop through active storms to generate the final plot boundaries (with correct aspect ratios),
# write out plotParams file with storm name, advisory number, center location, and all forecast
# date/times in plain english.
for($i=0;$i<$numStorms;$i++){
	
	$stormIDs[$i] =~ /(al|ep)(\d\d)(\d\d\d\d)/;	# parse storm id for:
	$stormBasin = $1;				#  - basin
	$stormNum = $2;					#  - storm number
	$stormYear = $3;				#  - year
	
	&parseADV("$stormIDs[$i]");

	if($singleStorm && $isSpecial ne "YES" && -e "$gridDir/tpcprb_$DTG"){
		print("\nFinal grid for this single storm has already been processed (and this is not a special advisory).. aborting processing\n");
		exit;
	}
	

	&adjustPlotBounds;		# Corrects plot boundaries for aspect ratio / maximum extent rules
		
	
	
	if($stormBasin ne "ep" && $stormBasin ne "al"){
		next;
	}
	
	&getTimes; 		# Compute date/time strings to use in the graphics
#	&getStormCenter;	# Get the storm center (for locating the storm center marker) - this is now retreived from "parse TCM"
	

	# stormInfo contains the text to write into plotParams - essentially everything 
	# that is needed to generate the final wind speed probability graphics.
	$stormInfo .= "CREATE PLOT\n";							# Identifies group of graphics (one for each storm)
	$stormInfo .= "$gridDir/$gridFileName\n";					# probability grid file name
	$stormInfo .= "$minLats[$i];$maxWests[$i];$maxLats[$i];$maxEasts[$i]\n";	# lower left and upper right corners (lat/lon) of plot
	print("$minLats[$i];$maxWests[$i];$maxLats[$i];$maxEasts[$i]\n");
	$stormInfo .= sprintf("%02.2f;%02.2f\n",$latDrawLon[$i],$lonDrawLat[$i]);	# latitude to draw longitudes at, longitude to draw latitudes at
	$stormInfo .= sprintf("%02.2f;%02.2f\n",$centerLat[$i],$centerLon[$i]);		# coordinates of storm center (lat/lon)
	$stormInfo .= "$stormBasin\n";							# storm basin (al, ep, cp)
	$stormInfo .= "$stormNum\n";							# storm number
	$stormInfo .= "$stormYear\n";							# year
	$stormInfo .= "$stormName[$i]\n";						# storm status and name (i.e. Hurricane Hugo)
	$stormInfo .= "$stormAdvisoryNumber[$i]\n";					# Advisory number, without leading 0s
	
	# Loop through all periods (intial through 120 hours by 6 hour increments)
	for(my $j=0;$j<=120;$j+=6){
		$stormInfo .= "$timeStamp[$i][$j]\n";					# Time stamp string for each period (i.e. 8 AM EDT on Tue Aug 8th)
	}
	
	$stormInfo .= "$advisoryTime[$i]\n";
}


# Write out plotParams file for use by wind speed probability image generation program
if(open(PLOTPARAMS,">$plotParams")){
	print PLOTPARAMS ($stormInfo);
	close(PLOTPARAMS);
}else{
	print("Could not open $plotParams for writing!  exiting.\n");
	exit;
}

system("/usr/bin/perl $binDir/windProbs_driver.pl"); #Generate plot

## End program
#######################################################################################



################################################################################
# Subroutine - processCornersFile :
# Opens the corners file, if it was available, and reads in the storm id's and
# plot corners (maximum extent of 2% probability contour for 34 knot winds)
################################################################################
sub processCornersFile{
		
	open(CORNERS, "$cornerFile") || die ("Could not open corners file: $cornerFile\n");
	@lines = <CORNERS>;
	
	$numStorms = 0;
	foreach $line (@lines){
		if($line =~ /(\w\w\d\d\d\d\d\d)\.fst/){
			push(@stormIDs,$1);
			$numStorms++;
		}elsif($line =~ /\s*(-?\d?\d?\d?\d\.\d).*;\s*(-?\d?\d?\d?\d\.\d).*;\s*(-?\d?\d?\d?\d\.\d).*;\s*(-?\d?\d?\d?\d\.\d).*/){
			$minLat  = $1 - 2;
			$maxWest = $2 - 2;
			$maxLat  = $3 + 1.5;
			$maxEast = $4 + 2;
			
			push(@minLats,$minLat);
			push(@maxLats,$maxLat);
			push(@maxWests,$maxWest);
			push(@maxEasts,$maxEast);		
		}
	}
}


################################################################################
# Subroutine - computeCorners :
# If the grid contains only one storm, use FORTRAN program (findcorners.exe)
# to get probability extent coordinates for setting map area (maximum extent
# of 2% probability countour for 34 knot winds)
################################################################################
sub computeCorners{
	if($DTG =~ /^\d\d(\d{6})(\d\d)$/){
		$gemDTG = $1 . "/" . "$2" . "00F120";  #format DTG in GEMPAK YYMMDDHH/Ffff
	}else{
		print("$DTG from file name not correct!  exiting...\n");
		exit;
	}
	
	
	$cornersLine = `$binDir/findcorners.exe $gridDir/$gridFileName $gemDTG`; # run FORTRAN program
#debug	print("$cornersLine");
	if($cornersLine =~ /corners:\s*(.*);\s*(.*);\s*(.*);\s*(.*)/){
		$minLat  = $1 - 2;
		$maxWest = $2 - 2;
		$maxLat  = $3 + 1.5;
		$maxEast = $4 + 2;
		print("\n$minLat $maxWest $maxLat $maxEast\n");
		
		push(@minLats,$minLat);
		push(@maxLats,$maxLat);
		push(@maxWests,$maxWest);
		push(@maxEasts,$maxEast);	
	}
}


################################################################################
# Subroutine - parseADV :
# Opens and parses the adv.txt storm info file for each storm.  The advisory number,
# storm category/name (e.g. Tropical Depression 10), advisory center position,
# and time zone are taken from the ADV.  If ADV cannot be found, use TCM/TCP/TCD 
################################################################################
sub parseADV{
	my $stormNumber = @_[0];
	my $advFileName = "$advDir/$stormNumber.adv.txt";
	
	my @advLines;	
	if(open(ADVFILE,"$advFileName")){
		@advLines = <ADVFILE>;

		close(ADVFILE);

	}else{
		print("\nCould not open ADV file $advFileName, attempting to use TCM/TCP/TCD for storm info\n");
		#exit;
		&parseTCM;
		&getTimesPublicDis;
		return;
	}

	my @advInfo;

	#put file into an array[i][j] such that i is the line, j is the element of that line (comma separated elements)
	foreach my $advLine (@advLines){
		chomp($advLine);
		my @elements = split(/,/,$advLine);
		push @advInfo, [ @elements ];
	}

	if($advInfo[0][0] = "STXT"){
		$stormName = $advInfo[0][1];
		$stormNumber = $advInfo[0][2];
	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}

	if($advInfo[1][0] = "BEST"){
		$stormName = $advInfo[1][1];
		$stormNumber = $advInfo[1][2];
		$bestStatus = $advInfo[1][3];
		$bestPressure = $advInfo[1][4];
		$bestMaxWind = $advInfo[1][5];
		$bestDTG = $advInfo[1][6];
		$bestLat = $advInfo[1][7]/10;
		$bestLon = $advInfo[1][8]/10;
		
		$advInfo[1][7] =~ /(\d+)(\w)/;
		$bestLat = $1/10;
		if($2 eq "S"){ $bestLat = -1 * $bestLat; }
		
		$advInfo[1][8] =~ /(\d+)(\w)/;
		$fstLon = $1/10;
		if($2 eq "W"){ $bestLon = -1 * $bestLon; }
	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}

	if($advInfo[2][0] = "FSTD"){
		$stormName = $advInfo[2][1];
		$stormNumber = $advInfo[2][2];
		$fstStatus = $advInfo[2][3];
		$fstPressure = $advInfo[2][4];
		$fstMaxWind = $advInfo[2][5];
		$fstDTG = $advInfo[2][6];
		$tau = $advInfo[2][7];
		
		$advInfo[2][8] =~ /(\d+)(\w)/;
		$fstLat = $1/10;
		if($2 eq "S"){ $fstLat = -1 * $fstLat; }
		
		$advInfo[2][9] =~ /(\d+)(\w)/;
		$fstLon = $1/10;
		if($2 eq "W"){ $fstLon = -1 * $fstLon; }
	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}

	if($advInfo[3][0] = "ADVD"){
		$stormName = $advInfo[3][1];
		$stormNumber = $advInfo[3][2];
		$advDTG = $advInfo[3][3];
		$tau = $advInfo[3][4];
		$advisoryNumber = $advInfo[3][5];
		$zone = $advInfo[3][6];
	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}

	if($advInfo[4][0] = "SPEC"){
		$isSpecial = $advInfo[4][1];
		$specialTime = $advInfo[4][2];
	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}

	if($advInfo[5][0] = "FINAL"){
		$isFinal = $advInfo[5][1];

	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}

	if($advInfo[6][0] = "GEOREF1"){

	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}

	if($advInfo[7][0] = "GEOREF2"){

	}else{
		print("\nadv file not in the proper format!\n");
		exit;
	}
	
	#Translate 2 letter storm status abbreviation 
	if( $fstStatus eq "TD" || $fstStatus eq "DB" ){ $status = "Tropical Depression"; }
	elsif( $fstStatus eq "TS" ){ $status = "Tropical Storm"; }
	elsif( $fstStatus eq "HU" ){ $status = "Hurricane"; }
	elsif( $fstStatus eq "SD" ){ $status = "Subtropical Depression"; }
	elsif( $fstStatus eq "SS" ){ $status = "Subtropical Storm"; }
	else{
		print("Could not identify tropical cyclone type, exiting");
		exit;
	}
	
	
	#compute 3 letter timezone abbreviation
	#if($isDST eq "YES"){
	#	$DST = "D";
	#}else{
	#	$DST = "S";
	#}	
	
	$timeZone[$i] = $zone;
	print("\nzone: $zone\n");	
#	print($timeZone[$i]);	
	if($timeZone[$i] eq "AST"){
                $timeOffset = -4;
        }elsif($timeZone[$i] eq "EST"){
                $timeOffset = -5;
        }elsif($timeZone[$i] eq "EDT"){
                $timeOffset = -4;
        }elsif($timeZone[$i] eq "CST"){
                $timeOffset = -6;
        }elsif($timeZone[$i] eq "CDT"){
                $timeOffset = -5;
        }elsif($timeZone[$i] eq "MST"){
                $timeOffset = -7;
        }elsif($timeZone[$i] eq "MDT"){
                $timeOffset = -6;
        }elsif($timeZone[$i] eq "PST"){
                $timeOffset = -8;
        }elsif($timeZone[$i] eq "PDT"){
                $timeOffset = -7;
        }
	
	if($stormBasin eq "ep" && $fstStatus eq "TD"){
		if($stormName =~ /^(one|two|three|four|five|six|seven|eight|nine|ten|eleven|twelve|twenty|thirty)/i){
			$stormName .= "-E";
		}
		if($stormName =~ /teen$/i){
			$stormName .= "-E";
		}
		
	}
		
	$centerLat[$i] = $bestLat;
	$centerLon[$i] = $bestLon;
	$stormName[$i] = uc($status) . " $stormName";
	$stormAdvisoryNumber[$i] = $advisoryNumber;
	
	
	
	#Set up time information
	#&getAdvisoryTime;
	$advisoryTime[$i] = "NOT COMPUTED (NOT USED)";
	

}



################################################################################
# Subroutine - parseTCM :
# Opens and parses the Forecast/Advisory for each storm.  The advisory number,
# storm category/name (e.g. Tropical Depression 10), and advisory center position
# are taken from the TCM (if no adv file exists)
################################################################################
sub parseTCM{
	opendir(MARDIR,"$marDir") || die "Could not open Forecast Advisory Directory: $marDir";
	@marAdvisories = readdir(MARDIR);
	close(MARDIR);
		foreach $marAdvisory (@marAdvisories){
			#if($marAdvisory =~ /^[al|ep]\d\d\d\d\d\d\.fstadv\.\d\d\d$/){			
			if($marAdvisory =~ /^$stormIDs[$i]\.fstadv\.\d\d\d$/){			
				open(MARADV,"$marDir/$marAdvisory") || die ("Could not open $marDir/$marAdvisory!!!");
				@lines = <MARADV>;
				$tempName = "";
				$tempAdvNum = "";
				$tempLat = "";
				$tempLon = "";
				foreach $line (@lines){
					if($line =~ /^(.*) FORECAST\/ADVISORY NUMBER\s+(\d?\d?\d)\s*/){
						$tempName = $1;
						$tempAdvNum = $2;
						#print "$fcstAdvDTG";
						$tempName =~ s/SPECIAL//;
					}elsif($line =~ /^AT $fcstAdvDTG CENTER WAS LOCATED NEAR\s+(\d?\d?\.\d)([N|S])\s+(\d?\d?\d?\.\d)([E|W])/){
						$stormName[$i] = $tempName;
						$stormAdvisoryNumber[$i] = $tempAdvNum;
						
						$tempLat = $1;
						$tempLon = $3;
						#print("\n$1 $2 $3 $4\n");
						
						if($2 eq "S"){
							$tempLat = -1 * $tempLat;
						}
						
						if($4 eq "W"){
							$tempLon = -1 * $tempLon;
						}			
						
						$centerLat[$i] = $tempLat;
						$centerLon[$i] = $tempLon;
					}
					
					# Center at advisory time
					#elsif($line =~/^REPEAT\.\.\.CENTER LOCATED NEAR\s+(\d?\d?\.\d)([N|S])\s+(\d?\d?\d?\.\d)([E|W])/){
					#	$tempLat = $1;
					#	$tempLon = $3;
					#	#print("\n$1 $2 $3 $4\n");
					#	
					#	if($2 eq "S"){
					#		$tempLat = -1 * $tempLat;
					#	}
					#	
					#	if($4 eq "W"){
					#		$tempLon = -1 * $tempLon;
					#	}											
					#}
				}
			}
		}	
}



################################################################################
# Subroutine - getAdvisoryTime :
# Gets the advisory time (as formatted in the public advisory)
# Since advisory time isn't actually used anywhere, don't compute it.
################################################################################
#sub getAdvisoryTime{
	
#	$DTG =~ /^(\d\d\d\d)(\d\d)(\d\d)(\d\d)$/;
#	my $year = $1 - 1900;
#	my $month = $2 - 1;
#	my $day = $3;
#	my $hour = $4 + 3;
#	my $minute = "00";
	
#	if($isSpecial eq "YES"){
#		$specialTime =~ /^(\d\d)(\d\d)?/;
#		$hour = $1;
#		print("It's a special!");
#		if($2) { $minute = $2; }
#	}
#		print("isDST: '$isDST'");
#	
#	
#	# Correlate the time zone offset from GMT/Z/UTC via the 3 letter time zone abbreviation
#	if($timeZone[$i] eq "AST"){
#		$timeOffset = -4;
#	}elsif($timeZone[$i] eq "EST"){
#		$timeOffset = -5;
#	}elsif($timeZone[$i] eq "EDT"){
#		$timeOffset = -4;
#	}elsif($timeZone[$i] eq "CST"){
#		$timeOffset = -6;
#	}elsif($timeZone[$i] eq "CDT"){
#		$timeOffset = -5;
#	}elsif($timeZone[$i] eq "MST"){
#		$timeOffset = -7;
#	}elsif($timeZone[$i] eq "MDT"){
#		$timeOffset = -6;
#	}elsif($timeZone[$i] eq "PST"){
#		$timeOffset = -8;
#	}elsif($timeZone[$i] eq "PDT"){
#		$timeOffset = -7;
#	}else{
#		$timeZone[$i] = "EDT";
#		if($stormBasin eq "ep"){
#			$timeZone[$i] = "PDT";
#		}
#	}
#	
#	
#	@months = ("Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec");  	# array of months for int to month string conversion
#	@weekDays = ("Sun","Mon","Tue","Wed","Thu","Fri","Sat");				# array of days for int to day string conversion
#		print("advisory time: $hour $timeOffset\n");
#	$stormTime = timegm(0,0,$hour,$day,$month,$year);	# get time (in seconds from epoch) in zulu for the advisory synoptic time
#	$stormTime = $stormTime + (3600*$timeOffset);		# adjust time to storm's time zone
#	
#
#	
#		($cursec,$curmin,$curhr,$curday,$curmon,$curyear,$curwd,$curyd,$isdst) = gmtime($stormTime);  #compute time variables for forecast time
#		$year = $curyear + 1900;
#		# Adjust 0-23 hour to 12 hours with AM/PM
#		if($curhr >= 12){
#			$AMorPM = "PM";
#			if($curhr > 12){
#				$curhr -= 12;
#			}
#		}else{
#			$AMorPM = "AM";
#		}		
#		if($curhr == 0){
#			$curhr = 12;
#		}
#		
#		# Figure out the correct suffix (1st, 2nd, 3rd, 17th, 23rd, etc.) to add to the day of the month.
#		if($curday >=10 && $curday <= 19){
#			$suffix = "th";
#		}elsif(($curday%10 <= 9 && $curday%10 >= 4) || $curday%10 == 0){
#			$suffix = "th";
#		}elsif($curday%10 == 1){
#			$suffix = "st";
#		}elsif($curday%10 == 2){
#			$suffix = "nd";
#		}elsif($curday%10 == 3){
#			$suffix = "rd";
#		}
#		
#		# Store the time stamp string: _8    AM     EST         Wed                Jun               14        
#		$advisoryTime[$i] = "$curhr $AMorPM $timeZone[$i] $weekDays[$curwd] $months[$curmon] $curday $year";
#	
#	
#	
#}

################################################################################
# Subroutine - adjustPlotBounds :
# Ensures plot boundaries are within the area that can be plotted, then expands
# the plot area to obtain the correct aspect ratio of the final graphic.  A
# degree of latitude is stretched in mercator coordinates, so the mercator
# equations are applied to convert lat/lon to mercator coordinates, compute the
# correct aspect ratio, and adjust the plot accordingly.  The "correct" aspect
# ratio results in GEMPAK leaving enough white space at the top and bottom
# of the graphic to write in the title, subtitle, logos, and legends.
################################################################################
sub adjustPlotBounds{
	$maxEastBound = 0;			# Do not cross the prime meridian (outside grid boundaries)
	$maxLatBound = 60;			# Do not cross 60 N (outside grid boundaries)
        
	
	# Make sure the initial center position is on the graphic.  The center position may 
	# not be on the graphic for a storm forecast to intensify in the later time periods.
	# Also, if there are no probabilities > 1% for the storm, inverse boundries are given...
	# in that case, this would set the initial boundaries to a 10 degree x 10 degree box
	# centered on the storm (or remnants).
	if( $centerLat[$i] > ($maxLats[$i] - 3) ){
		$maxLats[$i] = $centerLat[$i] + 3;
	}
	if( $centerLat[$i] < ($minLats[$i] + 3) ){
		$minLats[$i] = $centerLat[$i] - 3;
	}
	if( $centerLon[$i] > ($maxEasts[$i] - 3) ){
		$maxEasts[$i] = $centerLon[$i] + 3;
	}
	if( $centerLon[$i] < ($maxWests[$i] + 3) ){
		$maxWests[$i] = $centerLon[$i] - 3;
	}
	
	
	
	if($maxEasts[$i] > $maxEastBound){	# Make sure initial area is within the boundaries
		$maxEasts[$i] = $maxEastBound;  # if not, clip to boundaries
	}
	if($maxLats[$i] > $maxLatBound){	# Make sure initial area is within the boundaries
		$maxLats[$i] = $maxLatBound;	# if not, clip to boundaries
	}

	$lonDiff = $maxEasts[$i] - $maxWests[$i];	#width (in mercator coordinates) of the graphics area.  Longitude and mercator are the same units.	
	
	$maxLatCoordBound = rad2deg( log( tan( deg2rad( $maxLatBound ) ) + sec( deg2rad( $maxLatBound ) ) ) ); #Mercator y coordinate of 60N	
	$maxLatCoord = rad2deg( log( tan( deg2rad( $maxLats[$i] ) ) + sec( deg2rad( $maxLats[$i] ) ) ) );      # Maximum Longitude in Mercator coordinates
	$minLatCoord = rad2deg( log( tan( deg2rad( $minLats[$i] ) ) + sec( deg2rad( $minLats[$i] ) ) ) );      # Minumum Longitude in Mercator coordinates

	$latDiff = $maxLatCoord - $minLatCoord;		# Height (in mercator units) of the graphics area
	
	
	# Find the current aspect ratio and adjust it to be 
	# taller or wider to fit the target aspect ratio
	if(abs($lonDiff*2) > abs($latDiff*3)){						# If the image is too short	
	
		$latToAdd = ($lonDiff*2 - $latDiff*3)/3;				# Compute by how much
		$maxLatCoord += $latToAdd/2;						# add half the difference to the top
		$minLatCoord -= $latToAdd/2;						# and subtract half the difference from the bottom
		
		if($maxLatCoord > $maxLatCoordBound){					# if the area drawn goes north of 60
			$minLatCoord -= ($maxLatCoord - $maxLatCoordBound);		# shift the graphic down
			$maxLatCoord -= ($maxLatCoord - $maxLatCoordBound);		# by the difference
		}
		
		$maxLats[$i] = rad2deg( atan( sinh( deg2rad( $maxLatCoord ) ) ) );	# convert the northern extent mercator latitude back into latitude degrees
		$minLats[$i] = rad2deg( atan( sinh( deg2rad( $minLatCoord ) ) ) );	# convert the southern extent mercator latitude back into latitude degrees

	}else{									# If the image is too thin
	
		$lonToAdd = ($latDiff*3-$lonDiff*2)/2;				# Compute by how much
		$maxEasts[$i] += $lonToAdd/2;					# Add half the distance to the east
		$maxWests[$i] -= $lonToAdd/2;					# and half to the west
		
		if($maxEasts[$i] > $maxEastBound){				# If the graphic crosses the prime meridian
			$maxWests[$i] -= ($maxEasts[$i] - $maxEastBound);	# adjust westward by the amount
			$maxEasts[$i] -= ($maxEasts[$i] - $maxEastBound);	# it crosses the prime meridian
		}
	
	}
	
	$maxLats[$i] = sprintf("%.2f",$maxLats[$i]);	#Store northernmost plot latitude to two decimal places
	$minLats[$i] = sprintf("%.2f",$minLats[$i]);	#Store southernmost plot latitude to two decimal places	
	$maxWests[$i] = sprintf("%.2f",$maxWests[$i]);	#Store westernmost plot longitude to two decimal places
	$maxEasts[$i] = sprintf("%.2f",$maxEasts[$i]);	#Store easternmost plot longitude to two decimal places
	
	$width = $maxEasts[$i] - $maxWests[$i];
	$latDrawLon[$i] = $minLats[$i] + ($width * .0167);
	$lonDrawLat[$i] = $maxWests[$i] + ($width * .05);

}


################################################################################
# Subroutine - processNoCorners :
# In the event that there is no .corners file (like test runs on 2005 storms)
# figure out which storms are active by parsing the ATCF a-decks for OFCL
# lines with date/time groups matching the one specified at the command line.
# Then call getStormArea to compute a guess at the area the storm probabilities
# will cover.
################################################################################
sub processNoCorners{
	opendir(AIDDIR,$aidDir) || die "Could not open Forecast Advisory Directory: $aidDir\n";
	@adecks = readdir(AIDDIR);
	close(AIDDIR);
	
	foreach $adeck (@adecks){
	
		open(ADECK,"$aidDir/$adeck") || die ("Could not open adeck $aidDir/a$stormID.dat!!!\n");
			@adeckLines = <ADECK>;
		close(ADECK);
		foreach $line (@adeckLines){
			@adeckEntries = split /,\s*/,$line;
			if($adeckEntries[2] eq $DTG && $adeckEntries[4] eq "OFCL"){
				if($adeck =~ /^a((ep|al)(\d\d)\d\d\d\d)\.dat/){
					$stormID = $1;
					$stormNum = $3;
					if($stormNum > 0 && $stormNum < 80){
						push(@stormIDs,$stormID);
						$numStorms++;
						#print("Calling getStormArea($stormID,$DTG)");
						&getStormArea($stormID,$DTG);
						last;
					}
				}
			}
		}
	}
}



################################################################################
# Subroutine - getStormArea :
# In the event that there is no .corners file (like test runs on 2005 storms)
# compute a plot area that will likely contain all the probabilities data by
# adding the maximum forecast wind radii to the average forecast error+600nm
# and using the extent of the resulting area as plot bounds.
################################################################################
sub getStormArea{
	my ($stormID, $DTG) = @_;
	#print("$stormID $DTG\n");
	my @avgError = (10.4,18.4,42.2,75.3,106.9,137.8,202.4,235.7,310.2);
	#my @avgError = (30,60,120,180,300,400,500,500,500);
	my @lats;
	my @lons;
	my $maxRadii = 0;	
	
	$numForecasts = 0;
	foreach $line (@adeckLines){
		@adeckEntries = split /,\s*/,$line;
		if($adeckEntries[2] eq $DTG && $adeckEntries[4] eq "OFCL" && $adeckEntries[11] eq "34"){
			$adeckEntries[6] =~ /(\d*)(\w)/;
			$lat = $1/10;
			$adeckEntries[7] =~ /(\d*)(\w)/;
			$lon = $1;
			if($2 eq "W"){
				$lon = -1 * $lon/10;
			}
			#print("LON $lon");
			push(@lats,$lat);
			push(@lons,$lon);
			
			
			for(my $j = 13, $j<=16,$j++){
				if($adeckEntries[$j] > $maxRadii){
					$maxRadii = $adeckEntries[$j];
				}
			}
			$numForecasts++;
			#print "$adeckEntries[2] $adeckEntries[4] $lat $lon $adeckEntries[11] $adeckEntries[13] $adeckEntries[14] $adeckEntries[15] $adeckEntries[16]\n";
		}
	}
	
	$maxRadii = $maxRadii * 3;
	
	$minLat = 90;
	$maxLat = 0;
	$maxWest = 0;
	$maxEast = -360;
	for(my $j=0;$j<$numForecasts;$j++){
		$lonToAdd = ($avgError[$j] + $maxRadii)/(60*cos($lats[$i] * pi/180));
		$latToAdd = ($avgError[$j] + $maxRadii)/60;
		
		if(($lats[$j] - $latToAdd) < $minLat){
			$minLat = $lats[$j] - $latToAdd;
		}
		if(($lats[$j] + $latToAdd) > $maxLat){
			$maxLat = $lats[$j] + $latToAdd;
		}
		if(($lons[$j] + $lonToAdd) > $maxEast){
			$maxEast = $lons[$j] + $lonToAdd;
		}
		if(($lons[$j] - $lonToAdd) < $maxWest){
			$maxWest = $lons[$j] - $lonToAdd;
		}
	
	}
	push(@minLats,$minLat);
	push(@maxLats,$maxLat);
	push(@maxWests,$maxWest);
	push(@maxEasts,$maxEast);
}


################################################################################
# Subroutine - getTimes :
# Compute the date/time strings that will appear on the graphic
#		e.g.: 8 AM EST on Wed Jun 2nd
# Strings are computed for 0-120 hours at 6 hourly increments (21 strings)
################################################################################
sub getTimes{

	print("time offset: $timeOffset'");

	@months = ("Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec");  	# array of months for int to month string conversion
	@weekDays = ("Sun","Mon","Tue","Wed","Thu","Fri","Sat");				# array of days for int to day string conversion
	
	$DTG =~ /^(\d\d\d\d)(\d\d)(\d\d)(\d\d)$/;
	$year = $1 - 1900;
	$month = $2 - 1;
	$day = $3;
	$hour = $4;
	$stormTime = timegm(0,0,$hour,$day,$month,$year);	# get time (in seconds from epoch) in zulu for the advisory synoptic time
	$stormTime = $stormTime + (3600*$timeOffset);		# adjust time to storm's time zone
	

	# Loop through all 120 times, adding 6 hours and computing the string each time
	for(my $j=0;$j<=120;$j+=6){
		$timeStampTime = $stormTime + (3600*$j);	# add the number of hours the forecast is for
		($cursec,$curmin,$curhr,$curday,$curmon,$curyear,$curwd,$curyd,$isdst) = gmtime($timeStampTime);  #compute time variables for forecast time
		
		# Adjust 0-23 hour to 12 hours with AM/PM
		if($curhr >= 12){
			$AMorPM = "PM";
			if($curhr > 12){
				$curhr -= 12;
			}
		}else{
			$AMorPM = "AM";
		}		
		if($curhr == 0){
			$curhr = 12;
		}
		
		# Figure out the correct suffix (1st, 2nd, 3rd, 17th, 23rd, etc.) to add to the day of the month.
		if($curday >=10 && $curday <= 19){
			$suffix = "th";
		}elsif(($curday%10 <= 9 && $curday%10 >= 4) || $curday%10 == 0){
			$suffix = "th";
		}elsif($curday%10 == 1){
			$suffix = "st";
		}elsif($curday%10 == 2){
			$suffix = "nd";
		}elsif($curday%10 == 3){
			$suffix = "rd";
		}
		
		# Store the time stamp string: _8    AM     EST         Wed                Jun               14        
		$timeStamp[$i][$j] = "$curhr $AMorPM $timeZone[$i] $weekDays[$curwd] $months[$curmon] $curday";

	}

}

sub getTimesPublicDis{
# Open the public advisory for parsing to find the time zone.  EP storms do not have 
	# public advisories, so if there is no public get the time zone from the discussion
	$publicAdvisory = sprintf("$pubDir/$stormIDs[$i].public.%03d",$stormAdvisoryNumber[$i]);
	$discussion = sprintf("$disDir/$stormIDs[$i].discus.%03d",$stormAdvisoryNumber[$i]);
	$advisoryTime = "";
	
	if(-s $publicAdvisory){
		
		open(PUBLIC,"$publicAdvisory") || die ("Could not open $publicAdvisory");
		@publicLines = <PUBLIC>;
		close(PUBLIC);
		
		if($publicLines[5] =~ / (AST|EST|EDT|CST|CDT|MST|MDT|PST|PDT) /){
			$timeZone[$i] = $1;	
			@advisoryTimeSegments = split(/\s+/,$publicLines[5]);
		}elsif($publicLines[6] =~ / (AST|EST|EDT|CST|CDT|MST|MDT|PST|PDT) /){
			$timeZone[$i] = $1;	
			@advisoryTimeSegments = split(/\s+/,$publicLines[5]);
		}		
	}elsif(-s $discussion){
		open(DISCUSSION,"$discussion") || die ("Could not open $discussion");
		#print ("$publicAdvisory\n");

		@discussionLines = <DISCUSSION>;
		close(DISCUSSION);
		
		if($discussionLines[4] =~ / (AST|EST|EDT|CST|CDT|MST|MDT|PST|PDT) /){
			$timeZone[$i] = $1;
			@advisoryTimeSegments = split(/\s+/,$discussionLines[4]);
		}elsif($discussionLines[5] =~ / (AST|EST|EDT|CST|CDT|MST|MDT|PST|PDT) /){
			$timeZone[$i] = $1;
			@advisoryTimeSegments = split(/\s+/,$discussionLines[4]);
		}
		
	}
	

	@advisoryTimeSegments[0] =~ /^(\d?\d)(\d\d)$/;
	$advisoryTime[$i] = $1;
	if($2 ne "00"){
		$advisoryTime[$i] .= $2;
	}
	
	$advisoryTime[$i] .= " @advisoryTimeSegments[1] @advisoryTimeSegments[2] ";
	
	@advisoryTimeSegments[3] =~ /^(\w)(\w\w)$/;
	$advisoryTime[$i] .= $1 . lc($2) . " ";
	
	@advisoryTimeSegments[4] =~ /^(\w)(\w\w)$/;
	$advisoryTime[$i] .= $1 . lc($2) . " ";
	
	@advisoryTimeSegments[5] =~ /^(\d?)(\d)$/;
	if($1 eq "1" || $1 eq "2" || $1 eq "3"){
		$advisoryTime[$i] .= $1;
	}
	$advisoryTime[$i] .= $2;
		
	$advisoryTime[$i] .= " " . @advisoryTimeSegments[6];	
	
	
	# Set default time zones in case neither the public or discussion were found (possible)
	if($timeZone[$i] eq ""){
		$timeZone[$i] = "EDT";
		if($stormBasin eq "ep"){
			$timeZone[$i] = "PDT";
		}
	}
	
	# Correlate the time zone offset from GMT/Z/UTC via the 3 letter time zone abbreviation
	if($timeZone[$i] eq "AST"){
		$timeOffset = -4;
	}elsif($timeZone[$i] eq "EST"){
		$timeOffset = -5;
	}elsif($timeZone[$i] eq "EDT"){
		$timeOffset = -4;
	}elsif($timeZone[$i] eq "CST"){
		$timeOffset = -6;
	}elsif($timeZone[$i] eq "CDT"){
		$timeOffset = -5;
	}elsif($timeZone[$i] eq "MST"){
		$timeOffset = -7;
	}elsif($timeZone[$i] eq "MDT"){
		$timeOffset = -6;
	}elsif($timeZone[$i] eq "PST"){
		$timeOffset = -8;
	}elsif($timeZone[$i] eq "PDT"){
		$timeOffset = -7;
	}
}

