#! /usr/bin/perl

use 5.008;
use File::Copy;
use IO::File;
use Time::localtime;
use File::Path;    #Needed for directory creation

#--- Referenced Code
my $homeDir = $ENV{HOME};
$scriptsdir = "$homeDir/wgraph/wndprb/scripts"; # Directory on local box where scripts located.
require "$scriptsdir/windProbs_config.pl";	# configuration file
require $Config::windPlot;			# wind prob plotting routines
require $Config::generalUtils;			# general perl subroutines

#--- Directories
my $probDir = $Config::probDir;			# top level directory for probabilities
my $workDir = $Config::workDir;			# work directory
my $binDir = $Config::binDir;			# perl program directory
my $outputDir = $Config::outputDir;		# wind prob images storage directory
my $outputLogDir = $Config::outputLogDir;	# process log storage directory

mkpath($outputLogDir,0,0777);                   # Make the output and output log directories.

#--- Programs
my $getStormInfo = $Config::getStormInfo;	# parses ATCF files for storm info
my $gemClean = $Config::gemClean;		# remove left over message queues

my $DEBUG = 1;


#=-=-=-=-=-=--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
# Project Name: Wind Speed Probability Web Graphics
#
# Program Name: windProbs_driver.pl
#
# Called by: Crontab or command line
#
# Purpose: Manages the process of creating the wind speed proability graphics for 
#	   the public internet web site.
#
#----------------- DEPENDANCIES ------------------------------------
#
# Language:  Perl 5.8.x with Thread module
#
# External Programs:
#
#	generalUtils_v1.2.3 - Contains general utilities for 
#			     file and date/time functions
#	windProbs_plot.pl - Interface to GEMPAK plotting routines.
#	getStormInfo.pl -   Reads the ATCF forecast files and extracts 
#			    storms information which is written to
#			    an input file named plotParms. 
# Inputs:
#
#	plotParms - Text file containing formated storm data which is used
#		    to plot the wind probability graphics 
#	
# Outputs:
#
#	*.gif images of 34, 50 and 64 knot winds for f00, f12, f24, f36, f48, 
#				 f60, f72, f84, f96,  f108, f120		
#
#----------------- MAIN PARAMETERS ------------------------------------
# Variables:
#
# 	 $raStorms = []; 	- array of hashs and each hash contains 
#				  the information for one active storm.
#				  The information defined in the hash and 
#				  and example record is below.
#				  
#
# 	$rhStormInfo = { "gridFile"	=> "",
#			 "stormNumber" => "",
#	            	 "advisoryNumber" => "",
#		    	 "year" => "",
#		    	 "stormName" => "",
#		    	 "actadv" => "",
#		    	 "titles" => {},
#			 "boundaries" => ""  };
#
# 		Example Record: 
#			gridFile = /home/akrautkr/windProbs_methods/v2/tpcprbs/tpcprb_200509040
#	   		boundaries = 19;-78;53;-30
#	  		stormNumber = 14 
#	   		advisoryNumber = 012
#	  		year = 2005
#	   		stormName = HURRICANE MARIA
#	   		actadv = 12
#	   		titles = {"f00" = 2 AM AST Wed Dec 28,
#		     		"f12" = 2 PM AST Wed Dec 28,  ...
#		    	 	"f120" = 2 AM AST Mon Jan 2 }
#
#---------------- HISTORY ------------------------------------------
# Information:
#    Joint Hurricane Testbed/USWRP project
#    National Hurricane Center / Tropical Prediction Center
#    Department of Commerce - NOAA / NWS
#
# Category: Wind Speed Probability
#
# Written by: Alison Krautkramer 05/2005
# Modified: ARK 03/2006
#		Added documentation
#		Debugging
#		Logging 
# Modified: ARK 06/2006
#		Added abililty for the user to specify  which forecast hours
#               a wind speed probability plot will be created for a given storm.
#           Klein - 07/2008
#               Modified to run at HPC for 2008 season.
#
#=-=-=-=-=-=--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-


#-------------------------------------------
# 	*** Variable Declaration
#-------------------------------------------

#-- File handles

my $LOG = new IO::File;

#--- Files

my $inputFile = "plotParams";
my $logfile = "probGraph.log";
my $lockfile = "lock";
		
#-------------------------------------------
# 	*** Program Setup
#-------------------------------------------

#--- 3/20/2007 ARK
#--- Modification: Date no longer needed since this script no longer
#--- calls the getStormInfo.pl script which creates the plotParams input file.
#--- Get User Input 
#my $date = getInput();
#print ("Setting Date to current date/time to $date.\n");

#--- Check processes and setup work enviroment 

setupEnv(\$LOG);

generalUtils::header("STARTING WIND PROBABILITY GRAPHIC CODE", $LOG);


#--- Read data from input file and fill data structure 

my $raStorms = [];
getData($raStorms, $workDir, $inputFile, $LOG);

#--- Print data structure to log file

dumpStructure($raStorms, $LOG);

#-------------------------------------------
# 	*** Create Plot Objects
#-------------------------------------------

my $raPlotObjects = ();

foreach my $hStormInfo (@$raStorms) {

	#--- Check data validity

	if (checkData($hStormInfo, $LOG) == 1) {
	
		#--- Reformat data into gempak usuable format
		
		my @parsedElement = split (/;/, $$hStormInfo{"stormLocation"});
		$$hStormInfo{"gempakLoc"} = "$parsedElement[0] $parsedElement[1]";
	
		#--- Initialize grid type (All = all storms, ATCF = one storm)
		
		my @parsedElement = split(/\//, $$hStormInfo{"gridFile"});
		if ($parsedElement[-1] =~ /tpcprb_\d{10}/) {
			$$hStormInfo{'type'} = "ALL";
		}
		else {
			$$hStormInfo{'type'} = "ATCF";
		}
		
		#--- Create a unique subdirectory for gempak plotting
		
		my $uniqueDir = createUniqueDir ($workDir, $$hStormInfo{"basin"} . $$hStormInfo{"stormNumber"} ."_". $$hStormInfo{"advisoryNumber"});
		print ($LOG "Creating unique work directory, $uniqueDir, for processing.\n");
		
		#--- Create plot object
	
		push(@$raPlotObject, windProbs_plot::new($hStormInfo, $uniqueDir));
	
	}
	else {
	
		print( $LOG "#   Too many Errors: Not Creating Plot Object for " . $$hStormInfo{"stormNumber"} ."\n\n");
	
	}
			
}

#-------------------------------------------
# 	*** Plot Storms
#-------------------------------------------

#--- Create 120 hour plots for all storms

foreach my $plotObj (@$raPlotObject) {

	my $raFhour = ["120"];
	$plotObj->setFhour($raFhour);
	$plotObj->plot($LOG);
	$plotObj->storage($outputDir, $LOG);
}

#--- Create 00 - 114 plots for grids producted on the NCEP IBM which contain
#--- 		all of the active storms

my $num = 0;
foreach my $plotObj (@$raPlotObject) {

	my $raFhour = ["108", "96", "84", "72", "60", "48", "36", "24", "12", "00"];
	$plotObj->setFhour($raFhour);
		
	if (${$$raStorms[$num]}{'type'} eq "ALL") {
	
		$plotObj->plot($LOG);	

	}
	
	$plotObj->storage($outputDir, $LOG);
	$num++;
}

#foreach my $plotObj (@$raPlotObject) {
#	
#	my $raFhour = ["108", "96", "84", "72", "60", "48", "36", "24", "12", "00"];
#	$plotObj->setFhour($raFhour);
#	$plotObj->plot($LOG);
#	$plotObj->storage($outputDir, $LOG);	
#}


#-------------------------------------------
# 	*** Clean Up
#-------------------------------------------

#--- Remove intermediate files

close $LOG;
print ("$workDir/$logfile $outputLogDir/$logfile" . "_$date");
rename ("$workDir/$logfile", "$outputLogDir/$logfile" . "_$date");

#--- Remove Lock file
unlink ("$workDir/$lockfile");


#---------------------------------------------------------------------------------------------
#
#		SUBROUTINES
#
#---------------------------------------------------------------------------------------------




#--------------------------------------------------------
# Subroutine: getData
#
# Purpose: Parses active storm data text file which is produced by an 
#	   external shell script.  The parsed data is stored in a hash and
#	   added to the $raStorms array.
#
# Inputs: $raStorms - (array ref) - Reference to an empty array which will
#		be used to store pointers to hashes containing storm 
#		specific information extracted from a text file.
#	 $workDir - (string) - absolute path to the working directory
#	 $file - (string) - text file containing output from an external shell script
#	 $LOG - (file ref) - pointer to a LOG file
#
# Called by: main
#
# Modification: 3/20/2007 Added TYPE element to hash of information.  The type
#		will differentate between total storm grids and preliminary
#		storm specific grids created by the ATCF.
#--------------------------------------------------------
sub getData {

	#----------------------------------
	# *** Define Variables ***
	#----------------------------------
	
	my ($raStorms, $workDir, $file, $LOG) = @_;
	
	my $expectedLines = 31;
	my @fhours = ("00", "06", "12", "18", "24", "30", "36", "42", "48", "54", "60", "66", "72", "78", "84", "90", "96", "102", "108", "114", "120");
	
	#-------------------------------------------------
	# *** Open and read file contents ***
	#-------------------------------------------------
	
	my $INPUT = new IO::File;
	unless (generalUtils::openFile($workDir, $file, "read", \$INPUT) == 1) {
		
		fatal($workDir, $lockfile, "Problem reading $workDir/$file", $LOG);
	
	}
	
	my @lines = <$INPUT>;
	close $INPUT;
	
	
	#-------------------------------------------------
	# *** Fill Data Structure ***
	#------------------------------------------------
			
	for (my $num = 0; $num <= $#lines; $num++) {
		
		#--- Find beginning of new storm/instance
		
		if ($lines[$num] =~ /CREATE\s+PLOT/i  && $#lines >= ($num + $expectedLines)) {
			
			#-- Create Data Strucutre
		
			my $rhStormInfo = { "gridFile"	=> "",
					    "stormNumber" => "",
	            			    "advisoryNumber" => "",
		    			    "year" => "",
					    "basin" => "",
		    			    "stormName" => "",
		    			    "titles" => {},
					    "boundaries" => "",
					    "stormLocation" => "",
					    "latLonTextLoc" => "",
					     "advisoryTitle" => "",
					     "type"	=> "" 
					  };
					  
			addData($rhStormInfo, 'gridFile', \@lines, ($num + 1), 1, 0);		#--  Prob Grid
			addData($rhStormInfo, 'boundaries', \@lines, ($num + 2), 1, 0); 	#--  lat/lon boundaries
			addData($rhStormInfo, 'latLonTextLoc', \@lines, ($num + 3), 1, 0); 	#--  lat/lon lat/lon gempak text symbols
			addData($rhStormInfo, 'stormLocation', \@lines, ($num + 4), 1, 0); 	#--  lat/lon storm center
			addData($rhStormInfo, 'basin', \@lines, ($num + 5), 1, 1);		#--  storm basin
			addData($rhStormInfo, 'stormNumber', \@lines, ($num + 6), 1, 0);	#--  storm number
			addData($rhStormInfo, 'year', \@lines, ($num + 7), 1, 0);		#--- Year
			addData($rhStormInfo, 'stormName', \@lines, ($num + 8), 0, 1);		#--- Storm Name
			addData($rhStormInfo, 'advisoryNumber', \@lines, ($num + 9), 1, 0);	#--  advisory number
			
			#--- Line 9 - 18 - Date Strings
			
			my $lineCount = 10;
			
			foreach my $fhour (@fhours) {
				
				$$rhStormInfo{'titles'}{$fhour} = $lines[$num + $lineCount];
				chomp $$rhStormInfo{'titles'}{$fhour};
				$lineCount++;
	
			}
			
			$$rhStormInfo{'advisoryTitle'} = $lines[$num + $lineCount];
			chomp $$rhStormInfo{'advisoryTitle'};
			$lineCount++;
			
			$num = $num + $expectedLines;
			push (@$raStorms, $rhStormInfo);
			
			
		}
		elsif ( $#lines < ($num + $expectedLines) ) {
		
			print ($LOG "Input File Format Error (getData): Incorrect number of lines in $workDir/$file.\n");
			return -1;
		
		}
	
	}

	return 1;

}

#--------------------------------------------------------
# Subroutine: addData
#
# Purpose: Removes the end of line character from a value and
#	   stores it in the indicated hash.  If $space is set 
#	   to one then remove any extra spaces in the string.
#
# Inputs:  $rhData - (ref to hash) - stores storm information
#	   $aKey - (string) - represents a field within the hash
#	   $raLines - (ref to array) - ref to an array of strings
#	   $offset - (int) - index of item to extract from $raLines
#	   $space - (1 - yes, 0 - no) - indicates whether spaces should be 
#			removed from the string
#
# Called by: getData
#--------------------------------------------------------
sub addData {

	my ($rhData, $aKey, $raLines, $offset, $space, $capitalize) = @_;
	
	#--- Add data to hash
	
	$$rhData{$aKey} = $$raLines[$offset];
	
	#--- Remove newline character from string
	
	chomp $$rhData{$aKey};
			
	#--- Remove extra spaces from string if $space is set to 1		
			
	if ($space == 1) {
		
		$$rhData{$aKey} =~ s/\s+//g;
		
	}
	
	#--- Capitalize entry
	
	if ($capitalize == 1) {
	
		$$rhData{$aKey} =~ tr/a-z/A-Z/;
		
	}

}

#--------------------------------------------------------
# Subroutine: checkData
#
# Purpose: Checks the data added to the storm hash for validity.  This
#	   includes 
#		-test to make sure the grid exists and is fully transferred
#		-certain variables should only consist of digits
#		-the correct number elements is contains in the boundaries definition
#
# Inputs:  $rhStormInfo - (ref to hash) - Hash of storm information
#	   $LOG - (file ref) - pointer to a file for output
#	   
# Return: 1 - All check were satisfied
#	  -1 - Error found in at least one field
#
# Called by: main
#--------------------------------------------------------
sub checkData {

	my ($rhStormInfo, $LOG) = @_;
	
	#--- Check specific storm values for validity.  If any of the input
	#--- data is found to have errors then set the $returnCode variable
	#--- from 1 to -1.
	
	my $returnCode = 1;
	
	#--- Check the grid file name is valid and exists.
	
	if (checkGrid ($$rhStormInfo{'gridFile'}, $LOG) != 1) { $returnCode = -1};
	
	#--- Check the following fields only contains digits.
	
	if (generalUtils::checkInt($$rhStormInfo{'stormNumber'}, $LOG) != 1) { $returnCode = -1};
	if (generalUtils::checkInt($$rhStormInfo{'advisoryNumber'}, $LOG) != 1) { $returnCode = -1};
	if (generalUtils::checkInt($$rhStormInfo{'year'}, $LOG) != 1) { $returnCode = -1};
		
	#--- Check the 'boundaries' contains the correct number of digits separated by ';'
	
	if (checkBoundaries($$rhStormInfo{'boundaries'}, 4, $LOG) != 1) { $returnCode = -1};
	
	#--- Check the 'stormLocation' contains the correct number of digits separated by ';'
	
	if (checkBoundaries($$rhStormInfo{'stormLocation'}, 2, $LOG) != 1) { $returnCode = -1};
	
	#--- Check the 'latLonTextLoc' contains the correct number of digits separated by ';'
	
	if (checkBoundaries($$rhStormInfo{'latLonTextLoc'}, 2, $LOG) != 1) { $returnCode = -1};
	
	return $returnCode;
		
}

#--------------------------------------------------------
# Subroutine: checkBoundaries
#
# Purpose: Checks the storm boundary for validity.  This includes:
#		- string separated by semi-colons
#		- the correct number elements
#		- elements only contain either digits or '-' 
#
# Inputs:  $boundaries - (string) - formated storm boundaries
#	   $LOG - (file ref) - pointer to log file
#	   
# Return: 1 - All checks were satisfied
#	  -1 - Error found 
#
# Called by: checkData
#--------------------------------------------------------
sub checkBoundaries {

	my ($boundaries, $num, $LOG) = @_;
	
	#--- Separate the boundaries variable by the deliminator ';'
	
	my @parsedLine = split(/;/, $boundaries);
	
	#--- For each parsed element check that is only contains digits with the
	#--- possible exception of the negative sign.
		
	my $count = 0;
	
	foreach my $ele (@parsedLine) {
		
		#--- Check that the input number only contains digits
			
		if (generalUtils::checkFloat($ele, $LOG) == 1) {
			
			$count = $count + 1;
		}
		
	}
	
	#--- Print a error message if 4 number separated by ';' were not found
			
	unless ($count == $num) {
		
		print ($LOG "Error (main::checkData): In correct number of int used to define grid boundaries: $boundaries.\n");
		return -1;
		
	}
	
	return 1;

}

#--------------------------------------------------------
# Subroutine: checkGrid
#
# Purpose: Checks an absolute path to a grid file.  These checks include
#		- file exists
#		- file contains data
#		- file has been fully transferred to local computer
#
# Inputs: $gridFile - (string) - Absolute path to a grid file
#	  $LOG - (file handle) - Log file handle
#	   
# Return: -1 Error found in data format
#	  1 - OK
#
# Called by: checkData
#--------------------------------------------------------
sub checkGrid {

	my ($gridFile, $LOG) = @_;
	
	#--- Separate $gridFile into file and directory
	
	my $file;
	my $dir;
	
	unless (generalUtils::separateFile($gridFile, \$file, \$dir) == 1) {
	
		print ($LOG "Error (main::checkGrid): Problem occured parsing the grid file name: $gridFile.\n");
		return -1;
	
	}
	
	#--- Check the file exists and is none zero
	
	unless (generalUtils::isFile($dir, $file) == 1) {
		
		print ($LOG "Error (main::checkGrid): File either not found or has zero size: $gridFile.\n");
		return -1;
	}
	
	#--- Check Grid file is fully transfered for a total of 120
	#--- seconds before giving up and returning an error.
	
	unless ( generalUtils::checkFileTransfer($dir, $file, 120) == 1) {
	
		print ($LOG "Error (main::checkGrid): $gridFile either does not exist or is not fully transferred.\n");
		return -1;
	
	}

	return 1;
}

#--------------------------------------------------------
# Subroutine: dumpStructure
#
# Purpose: Prints the data structure of all storm information
#	   to a log file for debugging purposes.
#
# Input: $raStorms - (ref to array) - data structure containing storm information
#	 $LOG - (ref to file handle) - LOG file 
#
# Called by: main
#
# Return: -1 error condition encountered
#	  1 - OK
#--------------------------------------------------------
sub dumpStructure {

	#--- Define Variables 
	
	my ($raStorms, $LOG) = @_;
	
	my $count = 1;
	
	print ($LOG "START DUMPING STORM INFORMATION --------------------------\n\n");
	
	#--- Loop through each storm stored in the raStorms array
	
	foreach my $hEle (@$raStorms) {

		#--- Print the storm number to the log
		
		print ($LOG "------> STORM NUMBER " . $count . "\n\n");
	
		#--- Loop through all data values in each has associated with
		#--- the current storm
		
		foreach my $ele (keys %$hEle) {
			
			#--- Print out the text titles for each forecast hour
			
			if ($ele eq 'titles') {
		
				print ($LOG "         GRID TITLES ... \n");
		
				#--- Loop through forecast hours
		
				foreach my $fhour (keys %{$$hEle{$ele}}) {
				
					my $value = '';
					
					if (defined $$hEle{$ele}{$fhour}) {
					
						$value = $$hEle{$ele}{$fhour};
					
					}
				
					print ($LOG "           $fhour  $value \n");
				
				}
			
			}
			
			#--- Print out all non-title information
			
			else {
			
				#--- Captialize the hash key 
				
				my $capEle = $ele; 
				$capEle =~ tr/a-z/A-Z/;

				my $value = '';

				#--- Print the key and hash value to the log
				
				if (defined $$hEle{$ele}) {
					
					$value = $$hEle{$ele};
					
				}
			
				print ($LOG "         $capEle: $value \n");
				
			}	
		
		}
	
		print ($LOG "\n\n");
		$count++;
	
	}
	
	print ($LOG "END DUMPING STORM INFORMATION --------------------------\n\n");
	
	return 1;

}

#--------------------------------------------------------
# Subroutine: setupEnv
#
# Purpose: Sets up the environment needed to run this program including:
#
#		- Checking for the existance of the working directory.
#		 If the working directory does not exist then create
#		 create one.
#		- Open a log file
#
# Inputs: 	$workDir - (string) - Absolute path to the working directory
#		$logfile - (string) - log file name
#		$rLOG - (ref) - a reference to a file handle
#
# Called by: main
#
# Modifications: Lauers wrapper scripts will provide the plotParams input file.
#		As a result, removing the old file and recreating a new file is 
#		unnecessary.
#--------------------------------------------------------
sub setupEnv {

	#--- Define Variables 
	
	my ($rLOG) = @_;
	
	#--- Create the Work Directory
	
	unless (-d $workDir) {
	
		system ("mkdir $workDir") && die "Can't Open $workDir: $!";
	}
	
	#--- Create Lock Directory in workdir if one does not already exist

	createlock($workDir, $lockfile);
	
	#--- Clean up any left over message queues
	
	system("/usr/bin/perl $gemClean");

	#--- Remove previous $inputFile
	#
	#unlink ("$workDir/$inputFile");
	#
	#--- Run shell program to find a list of storms to process
	#--- and look for the resulting input file ($inputfile)
	#
	#system("/usr/bin/perl $getStormInfo $date");


	unless (-s "$workDir/$inputFile") {
	
		#-- code default to stdout .... since no log file
		fatal($workDir, $lockfile, "No active storms found.");
	
	}
	
	#--- Remove all old processing directories except leave the last one
	
	if ($DEBUG == 0) {
	
		opendir (WORK, $workDir);
		
		while (defined (my $name = readdir(WORK))) {
	
			if (-d "$workDir/$name" and $name =~ /\w{2}\d{2}_\d+_\d+/) {
				system ("/usr/bin/rm -rf $workDir/$name");
				#print ("delete $workDir/$name \n");
			}
		
		}
		closedir (WORK);
	}
	
	#--- Open Log file 
	
	unless ( generalUtils::openFile("$workDir", $logfile, "write", $rLOG) == 1 ) {
	
		print ("Error (main::setupEnv): $workDir/$logfile could not be opened for writing.\n");
	
	}

}

#--------------------------------------------------------
# Subroutine: createUniqueDir
#
# Purpose: Create a new sub-directory unique based on storm number and
#	   tbe date/time of processing.
#
# Input: $workDir - (string) - Absolute diretory path to a work directory
#	 $storm - (int) - ATCF storm number 
#	 $ logfile- (string) - lock file name
# 
# Called by: main
#
# Return: Absolute path name to the new unique sub-directory
#--------------------------------------------------------
sub createUniqueDir {

	#--- Define Variables 
	
	my ($workDir, $stormid, $logfile) = @_;
	
	#--- Create a unique subdirectory
	
	my $uniqueDir = ''; 
	
	#--- Create a directory name based on the current date/time.
	#--- Keep creating directory names until the directory is indeed unique. 
		
	do {
		my $tm = localtime;
		#$uniqueDir = ($tm->year + 1900) . ($tm->mon + 1) . $tm->mday . '_' . $tm->hour . $tm->min . $tm->sec;
		$uniqueDir = ($tm->year + 1900) . sprintf("%02s", ($tm->mon + 1)) . sprintf("%02s", ($tm->mday)) . '_' . sprintf("%02s", ($tm->hour)) . sprintf("%02s", ($tm->min)) .sprintf("%02s", ($tm->sec)); 
		$uniqueDir = '/' . $stormid . "_" . $uniqueDir;
		
	} while (-d "$workDir$uniqueDir");
	
	#--- Make directory
	
	system ("mkdir $workDir$uniqueDir") && die "Can't Open $workDir$uniqueDir: $!";
	
	return "$workDir$uniqueDir";

}

#--------------------------------------------------------
# Subroutine: createlock
#
# Purpose: Manages the creation of a lock file.  This includes checking
#	   if a lock file already exists.  
#		- If the lock file exist and is recent
#	  	  then this program will end. 
#		- If a lock file does not exist
#	  	 then a new file is creating.  
#		- If a lock file exists but is old then delete it and recreate
#		 a new lock file.
#
# Input: $workDir - (string) - Absolute diretory path to a work directory
#	 $lockfile - (string) - lock file name
# 
# Called by: main
#--------------------------------------------------------
sub createlock {

	
	#--- Define Variables 
	
	my ($workDir, $lockfile) = @_;
	
	#--- Check for existance of a lock file

	my $returnCode = generalUtils::isFile($workDir, $lockfile);

	if ($returnCode != -1) {
	
		#--- Check age of lock file ... if the lock file is older then some value
		#--- in minutes then assume this process died the last time it was
		#--- run.  Delete the lock file and run this program.
		
		if (generalUtils::oldFile($workDir, $lockfile, 10) == 0) {
	
			printf ("Error (main::createlock): Lock directory exists. Program already running. \n");
			exit;
		
		}
		
		#--- Remove old lock file
		
		unlink ("$workDir/$lockfile");
	
	}

	#--- Create a new lock file
	
	system ("touch $workDir/$lockfile");

}

#--------------------------------------------------------
# Subroutine: fatal
#
# Purpose: Fatal error managment.
#
# 	   Prints an error message to either the log or to stdout,
#	   removes the lock file and exits this script.
#
# Called by: main
#--------------------------------------------------------
sub fatal {

	#---- Define Variables
	
	my ($workDir, $lockfile, $message, $LOG) = @_;
	
	#--- Error message
	
	if (defined $LOG) {
		
		print ($LOG "Error: Ending Process .... $message\n");
		generalUtils::header("ERROR OCCURRED ENDING PROCESSING", $LOG);
		close $LOG;
	}
	else {
	
		print ("Error: Ending Process .... $message\n");
	
	}
	
	#--- Remove Lock File
	
	unlink ("$workDir/$lockfile");
	
	#--- End Process
	
	exit;
}


