package windProbs_plot;

use 5.008;
use File::Copy;
use IO::File;
use Thread;
use strict;
use warnings;

#--- scp

my $sendToWeb = $Config::sendToWeb;	# 1 -> yes, 0 -> no
my $remoteMachine1 = $Config::remoteMachine1;	#--- lnx131.ncep.noaa.gov (ip = 140.90.193.101)
my $remoteUser = $Config::remoteUser;

#--- Executables

my $convert = $Config::convert;
my $imageDir = $Config::imageDir;

#--- Graphics

my $noaaLogo = $Config::noaaLogo;
my $nwsLogo = $Config::nwsLogo;
my $clrBar = $Config::clrBar;
my $stormMarker = $Config::stormMarker;
my $noImage = $Config::noImage;	#--- ARK added for 0 - 108 prelim wind probs

#---- Flags

my $xvfb = 1;

my $gempakLogo = $Config::gempakLogo;	# 1 - gempak 0 - convert to add logos
my $gempakText = $Config::gempakText;	# 1 - gempak 0 - convert to add text boxes	
my $gempakClrBar = $Config::gempakClrBar;	# 1- gempak 0 - convert to add color bar


#=-=-=-=-=-=--=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
# Program Name: Wind Speed Probability Web Graphics
#
# Program Name: probPlot.pl 
#
# Language: Perl 5.8.x 
#
# Purpose: Manages programs used to create the *.gif image for NHC's public 
#	   ftp site.
#
#----------------- DEPENDANCIES ------------------------------------
#
# Language:  Perl 5.8.x with Thread module
#
# External Programs:
#
#	generalUtils_v1.2.4 - Contains general utilities for 
#			     file and date/time functions
#
#	prob.pl - Manages the overall creation of the probability graphics
#
#	convert - part of the ImageMagick software packages.  Convert is used
#		  to create the text which appears in the white space surrounding the
#		  image.
#
#	gempak - gempak is used to read the grid and plot the map, contour and storm marker
#
#------------------------INFORMATION------------------------------
#
# Joint Hurricane Testbed/USWRP project
# National Hurricane Center / Tropical Prediction Center
# Department of Commerce - NOAA / NWS
#
# Category: JHT - Plotting
#
# Written by: Alison Krautkramer 09/2005
# Modified by: ARK 04/2006
#		Added ability for caller to specify which forecast hours
#		a wind speed probability plot will be created for a given storm.
# Modified by: ARK 04/20/2006
#		New convert commands
#              Klein - 07/21/2008
#               Modify to run at HPC for 2008 season.  Use different convert 
#               commands as "annotate" not in RHEL3 OS versions.
#-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


my $rPlotSpeed = {
	
			"34Knots"	=> {    "smooth" => "SM5S(POP)",
						"pop" => "POP",
						"lowWind" => "39 mph",
						"windTitle" => "Tropical Storm Force" },
			
			"50Knots"	=> {	"smooth" => "SM5S(POPZ)",
						"pop" => "POPZ",
						"lowWind" => "58 mph",
						"windTitle" => "50-knot" },
			
			"64Knots"	=> {	"smooth" => "SM5S(POPF)",
						"pop" => "POPF",
						"lowWind" => "74 mph",
						"windTitle" => "Hurricane Force" }
	
};

my $rPlotType = {
			"smooth" => 	{
						"cint"	=>  "5;10;20;30",
						"fint"	=>  "5;10;20;30",
						"fline" =>  "0;23;22;21;5;0",
						"clrbar" =>  "32/H/LL/.14;.01/.36;.01/"	},
		
			"pop"	=> 	{
						"cint"	=> "40;50;60;70;80;90",
						"fint"	=> "40;50;60;70;80;90",
						"fline"	=> "0;19;18;17;15;14;30",
						"clrbar" => "32/H/LC/.655;.01/.45;.01/"	}
	
};
		
my $rPlotTitles = {	   
	   		"noaa"		=> {    "location" => "0.87;0.16",
						"textFile" => "noaa | 2.0 | C",
						"color"    => 6,
						"text"     => "1.2/11/1/111/1/c/sw"  },
			"nws"		=> {    "location" => "0.15;0.16",
						"textFile" => "nws | 2.0 | C",
						"color"    => 6,
						"text"     => "1.2/11/1/111/1/c/sw"  },
			"mainTitle" 	=> {    "location" => "0.5;0.96",
						"textFile" => "title.file",
						"imageFile" => "title.gif",
						"color"    => 32,
						"text"     => "1.3/23/1/221/1/c/hw"  },
			"subTitle"	=> {    "location" => "0.5;0.15",
						"textFile" => "title2.file",
						"imageFile" => "subTitle.gif",
						"color"    => 32,
						"text"     => "1.3/23/1/221/1/c/hw"  }
};

my @fhour = ("120", "114", "108", "102", "96", "90", "84", "78", "72", "66", "60", "54", "48", "42", "36", "30", "24", "18", "12", "06", "00");
my @speed = ("34Knots", "50Knots", "64Knots");


#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#
#
#		SUBROUTINES
#
#
#
#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


#--------------------------------------------------------
# Subroutine: new
#
# Purpose: Initialize a probPlot object with storm specific information.
#
# Inputs: $rhStormInfo - (ref to hash) - hash containing 
#			      storm specific information
#	  $dir - (string) - absolute path to a unique working directory
#
# Returns: plotPlot object
#
# Modification: 3/20/2007 Added TYPE element to hash of information.  The type
#		will differentate between total storm grids and preliminary
#		storm specific grids created by the ATCF.
#--------------------------------------------------------
sub new {

        #--- Define Variable 
	
	my ($rhStormInfo, $dir) = @_;

        #--- Define Data Structure 
	
	my $rPlot = {

		"device" 	=> "gf",
		"proj"		=> "MER//0;2;0;0",
		"filter"	=> "yes",
		"mapfile"	=> "hipowo.cia",
		"area"		=> $$rhStormInfo{'boundaries'},
		"advisoryNum"	=> $$rhStormInfo{'advisoryNumber'},
		"year"		=> $$rhStormInfo{'year'},
		"basin"		=> $$rhStormInfo{'basin'},
		"typeName"	=> $$rhStormInfo{'stormName'},
		"stormNum"	=> $$rhStormInfo{'stormNumber'},
		"gridFile"	=> $$rhStormInfo{'gridFile'},
		"titles"	=> $$rhStormInfo{'titles'},
		"advisoryTitle"	=> $$rhStormInfo{'advisoryTitle'},
		"stormLocation" => $$rhStormInfo{'gempakLoc'},
		"fhour" 	=> \@fhour,
		"dir"		=> $dir,
		"latLonTextLoc" => $$rhStormInfo{'latLonTextLoc'},
		"type"		=> $$rhStormInfo{'type'}
		
	};

	#--- Bless object
	
	bless $rPlot, "windProbs_plot";
	
	#--- Return created object
	
	return $rPlot;

}

#--------------------------------------------------------
# Subroutine:createImageName
#
# Purpose: Creates the unique name for the resulting 
#	   *.GIF result files. 
#
# Inputs:	$rPlot - (ref plot object) - probPlot object which
#			contains information used by the gempak routines
#		$speed - (string) - hash key indicating the wind radii being plotted
#			value = 34Knots, 50Knots, 64Knots
#		$hour - (string) - forecast hour
#
#--------------------------------------------------------
sub createImageName {

	#--------------------------------
        # *** Define Variable ***
	#---------------------------------

	my ($rPlot, $speed, $hour, $atype) = @_;
	
	my $stmnum = $rPlot->{"stormNum"};
	my $basin = $rPlot->{"basin"};
	my $knots = substr($speed, 0, 2);
	my $year = substr($rPlot->{"year"},2,2);
	my $stmadv = $rPlot->{"advisoryNum"};
	
	#--------------------------------
        # *** Create File Name ***
	#---------------------------------
	
	if ((defined $atype) and ($atype eq "internet")) {
		
		return "$basin$stmnum$year" . "_PROB$knots" . "_F$hour" . ".GIF"; 
	
	}
	
	$stmadv = sprintf ("%03s", $rPlot->{"advisoryNum"});
	$hour = sprintf ("%03s", $hour);
	
	#return "$basin$stmnum$year" . "_PROB$knots" . "_F$hour" . "_$stmadv.GIF"; 
	
	#--- ARK change 5/29/2007
	#return "$basin$stmnum$year" . "_PROB$knots" . "_$stmadv" . "_F$hour.GIF"; 
	return "$basin$stmnum" . $rPlot->{"year"} . "_PROB$knots" . "_$stmadv" . "_F$hour.GIF"; 
}

#--------------------------------------------------------
# Subroutine: plot
#
# Purpose: Manages the creation and rejoining of one thread
#	   of operation of each specific wind radii (34, 50, or 64)
#
# Inputs: $rPlot - (ref to probPlot object) - probPlot object which
#			contains information used by the gempak routines
#	  $dir - (string) - absolute path to a directory
#	  $LOG - (file handle) - pointer to log file
# 
#--------------------------------------------------------
sub storage {

	#--------------------------------
        # *** Define Variable ***
	#---------------------------------
	 
	my ($rPlot, $dir, $LOG) = @_;
	
	
	#--------------------------------
        # *** Store Graphics ***
	#---------------------------------
	
	if (defined $dir) {
		#--- Create storage dir if does not exist
		
		my $subDir = $rPlot->{basin} . $rPlot->{stormNum} . $rPlot->{year};
		
		unless (-d "$dir/$subDir") {
	
			system ("mkdir $dir/$subDir");
			print ($LOG "Create subdirectory $subDir for new graphics.\n");

		}
		
		#--- Copy graphics
		
		for my $aFhour (@{$$rPlot{fhour}}) {
		
			#--- Web Graphics
			for my $aSpeed (@speed) {
		
				my $gifName = createImageName($rPlot, $aSpeed, $aFhour);
				my $webName = createImageName($rPlot, $aSpeed, $aFhour, "internet");	
				
				#--- copy to storage directory
			
				#if (generalUtils::isFile($rPlot->{dir},$gifName)) {
				
				unless (generalUtils::isFile("$rPlot->{dir}/$aSpeed",$gifName) == 1) {
					system ("cp $imageDir/$noImage $rPlot->{dir}/$aSpeed/$gifName");	
					print "$gifName not found copy $noImage \n";
				}
				
					
				system ("cp $rPlot->{dir}/$aSpeed/$gifName $dir/$subDir/$gifName");
				print ($LOG "copy file $rPlot->{dir}/$aSpeed/$gifName to storage $dir/$subDir/$gifName\n");
					
				if ($sendToWeb == 1) {
				
					#--- scp to internet directory
						
					my $remoteDir = $rPlot->{basin};
					$remoteDir =~ tr/a-z/A-Z/;
					if ($remoteDir eq "AL") {
						$remoteDir = "AT" . $rPlot->{stormNum}
					}
					else {
						$remoteDir = $remoteDir . $rPlot->{stormNum}
					}
				
					system ("scp $rPlot->{dir}/$aSpeed/$gifName $remoteUser\@$remoteMachine1:~/$remoteDir/$webName\n");
					print ($LOG "scp $rPlot->{dir}/$aSpeed/$gifName $remoteUser\@$remoteMachine1:~/$remoteDir/$webName\n");
						
					#--- Code to archive wind probabilities
								
					if ($aFhour eq "120") {
						
						my $remoteDir = "/home/httpadm/docs/archive/$rPlot->{year}/graphics/$rPlot->{basin}$rPlot->{stormNum}";
						$remoteDir =~ tr/A-Z/a-z/;
						system ("scp $rPlot->{dir}/$aSpeed/$gifName $remoteUser\@$remoteMachine1:$remoteDir/$gifName\n");
						print ($LOG "scp $rPlot->{dir}/$aSpeed/$gifName $remoteUser\@$remoteMachine1:$remoteDir/$gifName\n");
						
					}
						
					
				}
				
			}
			#else {
			#	
			#	print ($LOG "Error (probPlot.pl::storage): $rPlot->{dir}/$gifName does not exist.\n");
			#	print ($LOG "Not copying file to $dir or scping file to $remoteMachine1. \n");
			#	
			#}
			
		}
			
	#}	
    
    }
	
}

#--------------------------------------------------------
# Subroutine: setFhour
#
# Purpose: Allows the caller to set the forecast hours that the wind
#	probability graphic will be produced.
#
# Inputs: $rPlot - (ref to probPlot object) - probPlot object which
#			contains information used by the gempak routines
#	  $raFhour - (ref to an array) - array containing a list
#			of forecast hours
# 
#--------------------------------------------------------
sub setFhour {	
	
	my ($rPlot, $raFhour) = @_;
	$$rPlot{"fhour"} = $raFhour;

}


#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#
#
#	Plot Routines
#
#
#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


#--------------------------------------------------------
# Subroutine: plot
#
# Purpose: Manages the creation and rejoining of one thread
#	   of operation of each specific wind radii (34, 50, or 64)
#
# Inputs: $rPlot - (ref to probPlot object) - probPlot object which
#			contains information used by the gempak routines
#	  $dir - (string) - absolute path to a directory
#	  $LOG - (file handle) - pointer to log file
# 
#--------------------------------------------------------
sub plot {

	#--------------------------------
        # *** Define Variable ***
	#---------------------------------- 
	
	my ($rPlot, $LOG) = @_;
	
	my @threads = ();
	my @logs = ();
	my $logfile = 'logfile';
	
	
	#------------------------------------------------------
        # ***  Create and Start One thread for each wind radii 
	#------------------------------------------------------
	
	for my $aSpeed (@speed) {
	
		#--- Create a sub directory
		
		my $speedDir = "$$rPlot{dir}/$aSpeed";
		system ("mkdir $speedDir");
		
		#--- Open Log file for the thread
		
		my $RADIILOG = new IO::File;
		unless ( generalUtils::openFile("$speedDir", $logfile, "write", \$RADIILOG) == 1 ) {
	
			print ($LOG "Error: $speedDir/$logfile could not be opened for writing.\n");
	
		}
		push (@logs, $RADIILOG);
		
		#--- Start Thread
		
		generalUtils::header("Starting thread for $rPlot->{typeName} at $aSpeed.\n", $RADIILOG);
		push (@threads, Thread->new(\&plotRadii, $rPlot, $aSpeed, $speedDir, $RADIILOG));
			
	}
	
	#------------------------------------------------------
        # ***  Join Threads and cleanup 
	#------------------------------------------------------
	
	foreach my $t (@threads) {
	
		$t->join();
		
	}
	
	#------------------------------------------------------
        # *** Cleanup 
	#------------------------------------------------------
	
	#--- Close logs
	
	foreach my $log (@logs) {

		close $log;
	
	}
	
	#--- Append to main log
	
	foreach my $aSpeed (@speed) {

		my $speedDir = "$$rPlot{dir}/$aSpeed";
		generalUtils::appendFile($speedDir, $logfile, $LOG);
		
	}

}

#--------------------------------------------------------
# Subroutine: plotRadii
#
# Purpose: Manages the creation of a shell script which calls
#	   GEMPAK programs and control the plotting of the wind
#	   speed probablity graphics. 
#
# Inputs:	$rPlot - (ref plot object) - probPlot object which
#			contains information used by the gempak routines
#		$speed - (string) - hash key indicating the wind radii being plotted
#			value = 34Knots, 50Knots, 64Knots
#		$dir - (string) - absolute path to the working directory
#		$LOG - (ref file) - log file handle
#
#--------------------------------------------------------
sub plotRadii {

        #--------------------------------
        # *** Define Variable ***
	#---------------------------------- 

	my ($rPlot, $speed, $dir, $LOG) = @_;
	
	#----------------------------------------------
	# *** Check GEMPAK flags
	#----------------------------------------------
	
	checkLogo($LOG);
	checkClrBar($LOG);
	
	#----------------------------------------------
	# *** Create Plot Text Files
	#----------------------------------------------
	
	createText($rPlot, $speed, $dir, $LOG);
	
	#----------------------------------------------
	# *** Create a shell script for GEMPAK commands
	#----------------------------------------------
	
	my $shellScript = "gempak.sh";
	print ($LOG "\n ----> Creating $shellScript. \n");
	my $GEMPAK = new IO::File;
	
       	unless ( generalUtils::openFile($dir, $shellScript, "write", \$GEMPAK) == 1) {
             	print ($LOG "Error (probPlot:createGrid): $dir/$shellScript could not be opened for writing.\n");
		return -1;
        }
	
	#--- Start Shell program and add start commands for virtual x server.
        print ($GEMPAK "#!/bin/ksh -f\n");
        print ($GEMPAK "dsp=`startXvfb_tpc 24`\n");
        print ($GEMPAK "if [ \$? -eq 0 ]; then\n");
        print ($GEMPAK "export DISPLAY=\$dsp\n");
        print ($GEMPAK "fi"); 
	
	#--- Add gempak commands
	gempakCmnds($rPlot, $speed, $dir, $GEMPAK, $LOG);
	
	#--- End shell program
	print  ($GEMPAK "exit 0\n");
	
	#--- Close file handle
       	close ($GEMPAK);
	
	#---------------------------------------------
        # *** Execute Shell script and cleanup
        #--------------------------------------------

	chmod (0744, "$dir/$shellScript");
        system ("cd $dir; $dir/$shellScript >& $dir/gempak.log");
	print ($LOG "Finished executing $dir/$shellScript. \n");
	
	#---------------------------------------------
        # *** Check image creation
        #--------------------------------------------
	
	my $found = 1;
	for my $aFhour (@{$$rPlot{fhour}}) {
		
		my $gifName = createImageName($rPlot, $speed, $aFhour);	
		unless  (generalUtils::isFile($dir, $gifName) == 1) {
			print ($LOG "Error (probPlot::plotRadii): $dir/$gifName not found or 0 size\n");
			$found = 0;
		}
	
	}	
	
        #---------------------------------------------
        # *** Add Logos/Text via convert
        #--------------------------------------------
	
	if (($gempakLogo == 0 or $gempakText == 0 or $gempakClrBar == 0) and ($found == 1)) {
	
		for my $aFhour (@{$$rPlot{fhour}}) {
		
			my $gifName = createImageName($rPlot, $speed, $aFhour);
			my $subTitleImage = $speed . "_" . $$rPlotTitles{'subTitle'}{'imageFile'};
			
			if ($gempakText == 0) {
				#my $subTitleImage = "f$aFhour" . "_" . $$rPlotTitles{'subTitle'}{'imageFile'};
				my $titleImage = "f$aFhour" . "_" . $$rPlotTitles{'mainTitle'}{'imageFile'};
				composite ($dir, $subTitleImage, $titleImage, $gifName);
			}
			else {	
				composite ($dir,  '', '', $gifName);
			}
			
		}
	
	}
        
	#---------------------------------------------
	# *** Cleanup
	#---------------------------------------------
		
  	unlink ("$dir/$shellScript");
        unlink ("$dir/gemglb.nts");
        unlink ("$dir/last.nts");
	
	#unlink ("$dir/$$rPlotTitles{'mainTitle'}{'imageFile'}");
	#system ("rm $dir/f*_" . $$rPlotTitles{'subTitle'}{'imageFile'});
	#system ("rm $dir/f*_" . $$rPlotTitles{'mainTitle'}{'imageFile'});	
		
}

#---------------------------------------------------------------------------------------
# Subroutine: gempakCmnds 
#
# Purpose: Adds GEMPAK commands to the gempak.sh shell program.
#	   These gempak commands control how the plot is created.
#
# Inputs:	$rPlot - (ref plot object) - probPlot object which
#			contains information used by the gempak routines
#		$speed - (string) - hash key indicating the wind radii being plotted
#			value = 34Knots, 50Knots, 64Knots
#		$dir - (string) - absolute path to the working directory
#		$GEMPAK - (ref file) - gempak.sh file handle
#		$LOG - (ref file) - log file handle
#
#---------------------------------------------------------------------------------------
sub gempakCmnds {

	#--------------------------------
        # *** Define Variable ***
	#---------------------------------- 
	
	my ($rPlot, $speed, $dir, $GEMPAK, $LOG) = @_;
	
	#--------------------------------
        # *** GEMPAK calls ***
	#---------------------------------- 
	
	#--- Set background to white
	gpcolor( $rPlot, "dummy.gif", "101=whi", $GEMPAK, $LOG );

	for my $aFhour (@{$$rPlot{fhour}}) {
		
		#--- Create image name
		my $gifName = createImageName($rPlot, $speed, $aFhour);	
		
		#--- Define Map
		gpmap ($rPlot, $gifName, $GEMPAK, $LOG );
		
		#--- Plot Contours
		
		#---------start gdplot3 method 
		
		#vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
		#--- ARK - 8/16/2007 Added fill variable and added another gdplot3 call in order to prevent color fills from obscuring 
		#--- the country and state names.
		#gdplot3 ($rPlot, $aFhour, $rPlotSpeed->{$speed}{smooth}, $rPlotType->{smooth}, $gifName, $rPlotTitles->{"mainTitle"}, "$dir/$$rPlotTitles{mainTitle}{textFile}" . $aFhour, "country", $GEMPAK, $LOG );
		#gdplot3 ($rPlot, $aFhour, $rPlotSpeed->{$speed}{pop}, $rPlotType->{pop}, $gifName, $rPlotTitles->{"subTitle"}, "$dir/$$rPlotTitles{subTitle}{textFile}" . $speed, "state", $GEMPAK, $LOG );
		gdplot3 ($rPlot, $aFhour, $rPlotSpeed->{$speed}{smooth}, $rPlotType->{smooth}, $gifName, $rPlotTitles->{"mainTitle"}, "$dir/$$rPlotTitles{mainTitle}{textFile}" . $aFhour, "", 1, $GEMPAK, $LOG );
		gdplot3 ($rPlot, $aFhour, $rPlotSpeed->{$speed}{pop}, $rPlotType->{pop}, $gifName, $rPlotTitles->{"subTitle"}, "$dir/$$rPlotTitles{subTitle}{textFile}" . $speed, "state", 1, $GEMPAK, $LOG );
		gdplot3 ($rPlot, $aFhour, $rPlotSpeed->{$speed}{smooth}, $rPlotType->{smooth}, $gifName, $rPlotTitles->{"mainTitle"}, "$dir/$$rPlotTitles{mainTitle}{textFile}" . $aFhour, "country", 0, $GEMPAK, $LOG );
		#^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^
		
		#---------end gdplot3 method 
		
		#---------start gdplot2 method 
		#gdplot2 ($rPlot, $aFhour, $rPlotSpeed->{$speed}{smooth}, $rPlotType->{smooth}, $gifName, $GEMPAK, $LOG );
		#gdplot2 ($rPlot, $aFhour, $rPlotSpeed->{$speed}{pop}, $rPlotType->{pop}, $gifName, $GEMPAK, $LOG );
		#gptext ($rPlot,$rPlotTitles->{mainTitle}, $$rPlotTitles{mainTitle}{textFile}, $gifName, "$dir/", $GEMPAK, $LOG );
		#gptext ($rPlot,$rPlotTitles->{subTitle}, $$rPlotTitles{subTitle}{textFile} . $aFhour, $gifName, "$dir/", $GEMPAK, $LOG );
		#--- end gdplot2method
		
		
		#--- Plot Logos with gempak if either gempakLogo is set to 1 or the image files are not found
		
		if ( $gempakLogo == 1 ) {

			gptext ($rPlot,$rPlotTitles->{noaa}, $$rPlotTitles{noaa}{textFile}, $gifName, " ", $GEMPAK, $LOG );
			gptext ($rPlot,$rPlotTitles->{nws}, $$rPlotTitles{nws}{textFile}, $gifName, " ", $GEMPAK, $LOG );
		
		}
		
		#-- Place storm marker on plot
		atest($rPlot, $GEMPAK, $LOG);	
		
	}
	
	#--- Set background to black
	#gpcolor( $rPlot, "dummy.gif", "101=bla", $GEMPAK, $LOG );
	
	#--- End gplt
        print  ($GEMPAK "gpend\n");
}


#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#
#
#	Check options
#
#
#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


#---------------------------------------------------------------------------------------
# Subroutine:checkLogo
#
# Purpose: If the user has indicated that the logos should be placed on 
#	   the graphic with convert then check if logo images exist.  If
#	   the images exist then return otherwise reset the gempakLogo flag
#	   to 1 so gempak will plot the logos.
#
# Inputs: $LOG - (ref file) - log file handle
#
#---------------------------------------------------------------------------------------
sub checkLogo {
	
	#--------------------------------
        # *** Define Variable ***
	#---------------------------------- 
	
	my ($LOG) = @_;
	
	#--------------------------------
        # *** Check logos ***
	#---------------------------------- 
	
	#--- if gempakLogo == 0 then check logo images exist
	#--- if images don't exist then set gempakLogo = 1
		
	if ($gempakLogo == 0 and (generalUtils::isFile($imageDir, $noaaLogo) != 1 or generalUtils::isFile($imageDir, $nwsLogo) != 1)) {
		
		print ($LOG "Error (probPlot::checkLogo): Either $imageDir/$noaaLogo or $imageDir/$nwsLogo not found.\n");
		print ($LOG "                             Plot logos with GEMPAK.\n");
		$gempakLogo = 1;	#set gempakLogo to default 1
		
	}
		
}

#---------------------------------------------------------------------------------------
# Subroutine:checkClrBar
#
# Purpose: If the user has indicated that the color bar should be placed on 
#	   the graphic with convert then check if the color bar image exists.  If
#	   the image exists then return otherwise reset the gempakClrBar flag
#	   to 1 so gempak will plot the color bar.
#
# Inputs: $LOG - (ref file) - log file handle
#
#---------------------------------------------------------------------------------------
sub checkClrBar {

	#--------------------------------
        # *** Define Variable ***
	#---------------------------------- 
	
	my ($LOG) = @_;
	
	#--------------------------------
        # *** Check color bar ***
	#---------------------------------- 
	
	#--- if gempakClrBar == 0 then check image exists
	#--- if image does not exist then set gempakclrBar = 1
	
	if ($gempakClrBar == 0 and (generalUtils::isFile($imageDir, $clrBar) != 1 )) {
		
		print ($LOG "Error (probPlot::checkClrBar): $imageDir/$clrBar is not found.\n");
		print ($LOG "                               Plot color bar with GEMPAK.\n");
		$gempakClrBar = 1;	#set gempakClrBar to default 1
		
	}
	
}


#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#
#
#	 TEXT SUBROUTINES
#
#
#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

#---------------------------------------------------------------------------------------
# Subroutine: createText
#
# Purpose: Manages the creation of the text which will be added
#	   to the gempak plot.  If the global variable $gempakText == 0
#	   then convert will be used to create *.gif images of the 
#	   text.  If the global variable $gempakText == 1 then
#	   ascii text file will be create and will be added to
#	   the grid using gempak.
#
# Inputs: $rPlot - (ref object) - probPlot object	
#	  $speed - (string) - indicates which wind speed is being plotted 
#			      (34, 50 or 64 knot) 
#	  $dir - (string) - absolute path for the title and subtitle files 
#	  $LOG - (ref file) - log file handle
#
#---------------------------------------------------------------------------------------
sub createText {
	
	#--------------------------------
        # *** Define Variable ***
	#---------------------------------- 
	
	my ($rPlot, $speed, $dir, $LOG) = @_;

	#----------------------------------------
        # *** Create title images for convert ***
	#-----------------------------------------
	
	if ($gempakText == 0) {
		
		#createTitleGraphic($rPlot, $$rPlotTitles{'mainTitle'}{'imageFile'}, $dir, $LOG);
		
		my $subTitleImage = $speed . "_" . $$rPlotTitles{'subTitle'}{'imageFile'};
		createSubTitleGraphic($rPlot, $subTitleImage, $dir, $speed, $LOG);
		
		for my $aFhour (@{$$rPlot{fhour}}) {
		
			my $titleImage = "f$aFhour" . "_" . $$rPlotTitles{'mainTitle'}{'imageFile'};
			createTitleGraphic($rPlot, $speed, $aFhour, $titleImage, $dir, $LOG);
			
			#my $subTitleImage = "f$aFhour" . "_" . $$rPlotTitles{'subTitle'}{'imageFile'};
			#createSubTitleGraphic($rPlot, $subTitleImage, $dir, $aFhour, $speed, $LOG);
		
		}
	}
		
	#----------------------------------------
        # *** Create text file for gempak ***
	#-----------------------------------------
	
	if ($gempakText == 1) {

		#createTitleFile($rPlot, $$rPlotTitles{'mainTitle'}{'textFile'}, $dir);
		#createSubTitleFiles($rPlot, $$rPlotTitles{'subTitle'}{'textFile'} . $aFhour, $dir, $aFhour, $speed);
		
		createSubTitleFile($rPlot, $$rPlotTitles{'subTitle'}{'textFile'} . $speed, $dir, $speed, $LOG);
	
		for my $aFhour (@{$$rPlot{fhour}}) {
	
			createTitleFile($rPlot, $speed, $aFhour, $$rPlotTitles{'mainTitle'}{'textFile'} . $aFhour, $dir, $LOG);
			
		}
	}
}

#--------------------------------------------------------
# Subroutine:createTitleFile
#
# Purpose: Creates an ascii text file containing the graphics title.
#	   This file will be added to the graphic using gempak.  
#
# Inputs: $rPlot - (ref object) - probPlot object	
#	  $file - (string) - sub title graphic file name 
#	  $dir - (string) - absolute path to the working directory	
#	  $speed - (string) - indicates the wind speed
#--------------------------------------------------------
sub createTitleFile {

	#--------------------------------
        # *** Define Variable ***
	#---------------------------------- 

	my ($rPlot, $speed, $fhour, $file, $dir, $LOG) = @_;
	
	my $rhTitles = \%{$$rPlot{'titles'}};
	
	#-------------------
	# *** Open File
	#-------------------
	
	my $INPUT = new IO::File;
	
	unless ( generalUtils::openFile($dir, $file, "write", \$INPUT) == 1) {
		print ("Error (probPlot:createTitleFile): $dir/$file could not be opened for writing.\n");
		return -1;
	}
	
	#-------------------
	# *** Add text 
	#-------------------
	
	my $title = $$rPlotSpeed{$speed}{'windTitle'} . " Wind Speed Probabilities\n";
	my $text = '';
	
	if ($fhour eq "00") {
		$text = "At $$rhTitles{'00'} ";
	}
	else {
	
		my $time = $fhour/24;
		if ($time > 1) {
			$time = $time . " days";
		}
		else {
			$time = $time . " day";
		}
		
		$text = "For the $fhour hours ($time) from $$rhTitles{'00'} to $$rhTitles{$fhour}\n";
	}
	
	print ($INPUT  "$title");
	print ($INPUT  "$text");
	
	#-------------------
	# *** Close File
	#-------------------
	
	close $INPUT;
	
}

#--------------------------------------------------------
# Subroutine:createSubTitleFile
#
# Purpose: Create an ascii text file containing the subtitle text.
#	   This ascii text file is used by gempak to add text to the image.
#
# Inputs: $rPlot - (ref object) - probPlot object	
#	  $file - (string) - sub title graphic file name 
#	  $dir - (string) - absolute path to the working directory	
#	  $fhour - (string) - indicates the forecast hour
#--------------------------------------------------------
sub createSubTitleFile {
	
	#--------------------------------
        # *** Define Variable ***
	#--------------------------------
		
	my ($rPlot, $file, $dir, $speed, $LOG ) = @_;	
	
	my $typename = $$rPlot{'typeName'};
	my $advisoryNum = $$rPlot{'advisoryNum'};
	my $rhTitles = \%{$$rPlot{'titles'}};
	my $year = $$rPlot{'year'};
	
	#-------------------
	# *** Open File
	#-------------------
	
	my $INPUT = new IO::File;
	
        unless ( generalUtils::openFile($dir, $file, "write", \$INPUT) == 1) {
              printf ("Error (probPlot:createSubTitleFiles): $dir/$file could not be opened for writing.\n");
	      return -1;
        }
	
	#-------------------
	# *** Add text 
	#-------------------
	
	my ($text1, $text2);
	
	my $lc_windTitle = $$rPlotSpeed{$speed}{'windTitle'};
	$lc_windTitle =~ tr/A-Z/a-z/;
	
	if ($speed eq "50Knots") {
	
		$text1 = "Probabilities of sustained (1-minute average) $lc_windTitle ($$rPlotSpeed{$speed}{'lowWind'}) or greater surface winds from all active tropical cyclones";
		$text2 = "indicates $typename center location at $$rhTitles{'00'} $year (Forecast/Advisory #$advisoryNum)";
	}
	else {
	
		$text1 = "Probabilities of sustained $lc_windTitle surface winds (1-minute average of $$rPlotSpeed{$speed}{'lowWind'} or greater) from all active tropical cyclones";
		$text2 = "indicates $typename center location at $$rhTitles{'00'} $year (Forecast/Advisory #$advisoryNum)";
	}
	
	print ($INPUT "$text1\n");
	print ($INPUT "$text2\n");
	
	#-------------------
	# *** Close File
	#-------------------
	
	close $INPUT;
	
}

#-------------------------------------------------------------------------------------
# Subroutine:createTitleGraphic
#
# Purpose: Creates an image of the title text. 
#
# Inputs: $rPlot - (ref object) - probPlot object	
#	  $file - (string) - sub title graphic file name 
#	  $dir - (string) - absolute path to the working directory	
#	  $speed - (string) - indicates the wind speed
#	  $LOG - (ref file handle) - log file handle
#--------------------------------------------------------------------------------------
sub createTitleGraphic {
		
	#--------------------------------
        # *** Define Variable ***
        #--------------------------------

	my ($rPlot, $speed, $fhour, $file, $dir, $LOG) = @_;
	
	my $rhTitles = \%{$$rPlot{'titles'}};
	
	#--------------------------------------
        # *** Initalize Text  ***
        #--------------------------------------
	
	my $pointsize = 28;
	my $title = $$rPlotSpeed{$speed}{'windTitle'} . " Wind Speed Probabilities\n";
	
	if ($$rPlot{'type'} eq "ATCF") {
		 $title = "PRELIMINARY (SINGLE STORM) " . $$rPlotSpeed{$speed}{'windTitle'} . " Wind Speed Probabilities\n";
		 $pointsize = 20
	}
	
	my $text = '';
	
	if ($fhour eq "00") {
		$text = "At $$rhTitles{'00'} ";
	}
	else {
	
		my $time = $fhour/24;
		if ($time > 1) {
			$time = $time . " days";
		}
		else {
			$time = $time . " day";
		}
		
		$text = "For the $fhour hours ($time) from $$rhTitles{'00'} to $$rhTitles{$fhour} ";
	}
	
	#--------------------------------------
        # *** Create Main Title ***
        #--------------------------------------
	
	my $size = "1118x50";
	
	if ($xvfb == 1) {
		$size = "895x49";
	}
	
	#--- The -colors option restricts the number of colors used by the text to only 8 colors.
	#--- The +dither option turns off the dithering algorithm and results in unsmeared text.
	
#	system ("$convert -colors 8 -size $size xc:white -font Helevetica \\
#		-pointsize $pointsize -gravity center +dither \\
#		-fill black \\
#		-annotate 0x0+0+4 \"$title\" \\
#		-pointsize 18 \\
#		-annotate 0x0+0+9 \"$text\" \\
#		$dir/$file");


        #--- Note that "annotate" is not on RHEL3 machines...so need to use a modified convert
        #--- command.  Products won't look exactly the same...but it's acceptable.

        system ("$convert -colors 8 -size $size xc:white -font Helevetica \\
                -pointsize $pointsize -gravity center +dither \\
                -fill black \\
                -draw \"text 0,4 '$title'\" \\
                -pointsize 18 \\
                -draw \"text 0,9 '$text'\" \\
                $dir/$file");

	print ($LOG "Created subtitle graphic containing text: $title $text \n");
	
}

#-----------------------------------------------------------------------------------
# Subroutine:createSubTitleGraphic
#
# Purpose: Creates a *.gif file containing the sub-title text within
#	   white box.
#
# Inputs: $rPlot - (ref object) - probPlot object	
#	  $file - (string) - sub title graphic file name 
#	  $dir - (string) - absolute path to the working directory	
#	  $fhour - (string) - indicates the forecast hour
#	  $speed - (string) - indicates the wind speed
#	  $LOG - (ref file handle) - log file handle
#-----------------------------------------------------------------------------------
sub createSubTitleGraphic {
		
	#-----------------------------------
        # *** Define Variable ***
        #----------------------------------
		
	my ($rPlot, $file, $dir, $speed, $LOG) = @_;	
		
	my $typename =  $$rPlot{'typeName'};
	my $advisoryNum = $$rPlot{'advisoryNum'};
	my $rhTitles = \%{$$rPlot{'titles'}};
	my $year = $$rPlot{'year'};
		
	#-------------------------------------
        # *** Initalize Text ***
        #-------------------------------------
	
	my ($text1, $text2);
	
	my $lc_windTitle = $$rPlotSpeed{$speed}{'windTitle'};
	$lc_windTitle =~ tr/A-Z/a-z/;
	
	if ($speed eq "50Knots") {
	
		#$text1 = "Probabilities of sustained (1-minute average) $lc_windTitle ($$rPlotSpeed{$speed}{'lowWind'}) or greater surface winds from all active tropical cyclones";
		if ($$rPlot{'type'} eq "ATCF") {
			$text1 = "Probability of 1-minute average $lc_windTitle ($$rPlotSpeed{$speed}{'lowWind'}) or greater surface winds";
		}
		else {
			$text1 = "Probability of 1-minute average $lc_windTitle ($$rPlotSpeed{$speed}{'lowWind'}) or greater surface winds from all tropical cyclones";
		}
		$text2 = "indicates $typename center location at $$rhTitles{'00'} $year (Forecast/Advisory #$advisoryNum)";
	}
	else {
	
		#$text1 = "Probabilities of sustained $lc_windTitle surface winds (1-minute average of $$rPlotSpeed{$speed}{'lowWind'} or greater) from all active tropical cyclones";
		if ($$rPlot{'type'} eq "ATCF") {
			$text1 = "Probability of sustained $lc_windTitle surface winds (1-minute average of $$rPlotSpeed{$speed}{'lowWind'} or greater)";
		}
		else {
			$text1 = "Probability of $lc_windTitle surface winds (1-minute average >= $$rPlotSpeed{$speed}{'lowWind'}) from all tropical cyclones";
		}
		$text2 = "indicates $typename center location at $$rhTitles{'00'} $year (Forecast/Advisory #$advisoryNum)";
	}
	
	#-------------------------------------
        # *** Create Text Box ***
        #-------------------------------------
	
	my $size = "1118x25";
	
	if ($xvfb == 1) {
		$size = "895x22";
	}
	
	#--- Decrease the font size if $text2 is too long
	
	my $pointsize = 15;
	$_ = $text2;
	my $charCount = tr/a-zA-Z -()#0-9\///; 
	if ($charCount >= 113) {
		$pointsize = 14;
	}
		 
        #stormMarker_2.gif label:"indicates SUBTROPICAL DEPRESSION TWENTY-SEVEN-E center location at 11 PM EDT on Www MMM 88 2000 (Advisory #00)" +append test.gif
		 
	# ask Lauer.  Creates the "marker/storm name/advisory number line.  Expands the canvas and then crops to the right size to center (hack)
#	system ("$convert -gravity center $imageDir/$stormMarker \\
#	         -colors 8 +dither -fill black \\
#	         -font Helvetica -pointsize $pointsize \\
#		 label:'$text2' +append -bordercolor white -border 400x100 \\
#		 -crop 895x20+0+0 +repage \\
#		 $dir/sub_$file");		 

        #--- Note that "annotate" is not on RHEL3 machines...so need to use a modified convert
        #--- command.  Products won't look exactly the same...but it's acceptable.

        # The next series of steps are a workaround for versions of imagemagick prior to 6.0.4 where there is no annotate feature.
        # First copy the diamond-shape storm marker image to a temp file...then to center it, append some white space to the beginning of it.
        # Then append the $text2 variable from above to complete that line.
        # To create the line above this, append some white space above first

        my $tempfile = "$dir/spacer.gif";
        system ("cp $imageDir/$stormMarker $tempfile");

        system ("$convert -size 10x22 xc:white -gravity center $tempfile +append $tempfile");
    
        system ("$convert -gravity center $tempfile \\
                -colors 8 +dither -fill black \\
                -font Helvetica -pointsize $pointsize \\
                label:'$text2' +append $dir/sub_$file");

        # To create the other line of the subtitle, append white space above the text2 line.
        # And then append the text1 line.      

        system ("$convert -colors 8 -background white -size $size xc:white -font Helvetica \\
                 $dir/sub_$file -append $dir/sub_$file");

        system ("$convert -gravity center +dither $dir/sub_$file  \\
                 -pointsize 15 \\
                 -draw \"text 15,18 '$text1'\" $dir/$file");
 
        # Remove the intermediate file and marker temp file.

        unlink("$dir/sub_$file");
        unlink("$tempfile");

        #--- The -colors option restricts the number of colors used by the text to only 8 colors.
	#--- The +dither option turns off the dithering algorithm and results in unsmeared text.
	
		
#	system ("$convert -colors 8 -size $size xc:white -font Helvetica  \\
#	         -pointsize 19 -gravity center +dither \\
#		 -fill black \\
#		 -annotate 0x0+0-2  \"$text1\" \\
#		 $dir/sub_$file -append \\
#		 $dir/$file");
		 
#		 unlink("$dir/sub_$file");	 
        	
		 
	print ($LOG "Created subtitle graphic containing text: $text1 $text2 \n");
	
}

#--------------------------------------------------------------------------------------
# Subroutine:composite
#
# Purpose: Overlays separate images over each other
#
# Inputs: $dir - (string) - working directory which contains 
#			   the title and subtitle graphics
#	  $subTitleGraphic - (string) - file name of the bottom text graphic
#	  $titleGraphic - (string) - file name of the top text graphic
#	  $windProbGraphic - (string) - file name of the final wind prob
#				graphic 
#---------------------------------------------------------------------------------------
sub composite {

	my ($dir, $subTitleGraphic, $titleGraphic, $windProbGraphic) = @_;

	my ($noaaLoc, $nwsLoc, $titleLoc, $subTitleLoc, $clrBarLoc, $stormMrkr);
	
	if ($xvfb == 0) {
	
		#--- ARK change made to switch logo positions on graphic
		#$nwsLoc = "+0+0";
		#$noaaLoc = "+1060+0";
		$nwsLoc = "+1060+0";
		$noaaLoc =  "+0+0";
		$titleLoc = "+0+0";
		$subTitleLoc ="+0+782";
		$clrBarLoc = "+228+815";
		$stormMrkr = "+345+31";
		
	}	
	else {
		
		#--- ARK change made to switch logo positions on graphic
                #--- Klein/HPC...slight change to top title location to get it away from the extreme 
                #    top of the graphic.
		#$nwsLoc = "+2+0";
		#$noaaLoc = "+845+0";
		$nwsLoc = "+845+0";
		$noaaLoc =  "+2+0";
		$titleLoc = "+0+2";
		$subTitleLoc ="+0+645";
		$clrBarLoc = "+130+690";
		$stormMrkr = "+20+670";

	}
	
	my $text = "";
	
	#--- Add the text images
	if ($gempakText == 0) {
	
		$text = $text . " -page $titleLoc '$dir/$titleGraphic' -page $subTitleLoc '$dir/$subTitleGraphic' ";
	}
	
	#--- Add the noaa/nws logos
	if ($gempakLogo == 0) {
	
		$text = $text .  " -page $noaaLoc '$imageDir/$noaaLogo' -page $nwsLoc '$imageDir/$nwsLogo' ";
	}
	
	#--- Add the color bar
	if ($gempakClrBar == 0) {
	
		$text = $text . " -page $clrBarLoc '$imageDir/$clrBar' ";
	}
	
	#--- Add the storm marker next to the title graphic
	#$text = $text . " -page $stormMrkr '$imageDir/$stormMarker' ";

	#--- Call convert to create graphics
	system ("$convert $dir/$windProbGraphic $text -flatten $dir/$windProbGraphic");
			
}


#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-
#
#
#		GEMPAK PLOTTING ROUTINES
#
#
#=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=


#--------------------------------------------------------------------------------
# GPMAP - Sets the map variables 
#
#	INPUT PARAMETERS
#	MAP       Map color/dash/width/filter flag  
# 	MSCALE    fgc;bgc;mask/units/lat;hide/valu  
# 	GAREA     Graphics area                   
# 	PROJ      Map projection/angles/margins|dr  
# 	SATFIL    Satellite image filename(s)
# 	RADFIL    Radar image filename(s)
# 	IMCBAR    Color/ornt/anch/x;y/ln;wd/freq
# 	LATLON    Line color/dash/width/freq/inc/l
# 	PANEL     Panel loc/color/dash/width/regn   
# 	TITLE     Title color/line/title            
# 	TEXT      Size/fnt/wdth/brdr/N-rot/just/hw  
# 	CLEAR     Clear screen flag                 
# 	DEVICE    Device|name|x size;y size|color   
# 	LUTFIL    Enhancement lookup table filenam
# 	STNPLT    Txtc/txt attr|marker attr|stnfil
# 	VGFILE    Vgfile | scale file | attribute
# 	AFOSFL    AFOS Graphics File
# 	AWPSFL    AWIPS Graphics File
#	 LINE      Color/type/width/label/smth/fltr  
# 	WATCH     End time|Wtch clrs|Wtch Tm;Statu
# 	WARN      End time|TS;TN;FF clrs|Tm|Lb|Out
# 	HRCN      End time|colors|syms|Tm|Lb|Mt|Qw
# 	ISIG      End time|colors|Sym|Tm|Id|Mv|Fl
# 	LTNG      End time|time ints/colors|marker
# 	ATCF      Time|colors|models|Tm|Id|Mv|Mkr|
# 	AIRM      Airmet Plotting Attributes
# 	NCON      NCON attributes
# 	CSIG      End time|0_Hr;1_Hr;2_Hr;OL clrs|
# 	SVRL      End time|SVRL clrs|Tm|Lb|Outline
# 	BND       Bnd name/color/fillsiz/fillpat/f
# 	TCMG      End time|colors|center
# 	QSCT      QuickScat Plotting Attributes
# 	WSTM      End time|WN;WT;AD clrs|Tm|Lb|Out
# 	WOU       End time|Cnty bnds clrs|Tm|Lb|Wt
# 	WCN       End time|Cnty bnds clrs|Tm|Lb|Wt
#	
# Modifications: 3/20/2007 ARK
#		   - Added \$MAPFIL = $rPlot->{mapfile} in to plot the high resolution coast lines
#		   - Set LATLON = 0 to plot the lat/lon lines above the storm
#
#--------------------------------------------------------------------------------------
sub gpmap {

	#--- Define local variables

        my ($rPlot, $giffile, $GEMPAK, $LOG) = @_;
 	
print ($LOG "         Plotting lat/lon lines and map projection for $rPlot->{area}.\n");	
print $GEMPAK <<END_OF_TEXT;
	
gpmap <<EOF;
\$MAPFIL  = $rPlot->{mapfile}
MAP      = 32//1 
MSCALE   = 0
GAREA    = $rPlot->{area}
PROJ     = $rPlot->{proj}
SATFIL   =  
RADFIL   =  
IMCBAR   =  
LATLON   = 0
PANEL    = 0
TITLE    = 0
TEXT     = 1/21/2/111/1/c/hw
CLEAR    = no 	
DEVICE   = $rPlot->{device} | $giffile
LUTFIL   = none
STNPLT   =
VGFILE   =  
AFOSFL   =  
AWPSFL   =  
LINE     = 31/1/1/1
WATCH    =  
WARN     =  
HRCN     =  
ISIG     =  
LTNG     =  
ATCF     =  
AIRM     =  
NCON     =  
CSIG     =  
SVRL     =  
BND      = bg/6 + world_bnds/10  + lakes/6
TCMG     =  
QSCT     =  
WSTM     =  
WOU      =  
WCN      = 

list

run

EOF

END_OF_TEXT

}

#--------------------------------------------------------------------------------
#GPCOLOR - changes the colors on a color device
#
#	INPUT PARAMETERS
#	COLORS    Color list                        
# 	DEVICE    Device|name|x size;y size|color  
#	
#-----------------------------------------------------------------------------
sub gpcolor{

  #--- Define local variables
  
  my ($rPlot, $giffile, $color, $GEMPAK, $LOG) = @_;

print ($LOG "         Changing background color of plot to white.\n"); 	
print $GEMPAK <<END_OF_TEXT;

gpcolor <<EOF;
\\\$mapfil = hipowo.cia
colors 	= $color
device 	= $rPlot->{device} | $giffile
clear  = yes

run

EOF

END_OF_TEXT

}

#--------------------------------------------------------------------------------
#GDPLOT3 - contour and fill data
#
#	INPUT PARAMETERS
#	GDFILE   Grid file
#	GDATTIM  Grid date/time
#	GLEVEL   Grid level
#	GVCORD   Grid vertical coordinate
#	PANEL    Panel loc/color/dash/width/regn
#	SKIP     Skip_cntr/skip_plt_x;skip_plt_y
#	SCALE    Scalar scale / vector scale
#	GDPFUN   Scalar grid or vector grid function
#	TYPE     GDPLOT2 function processing type: C/L/I/F/X/A/B/S/G/P
#	CONTUR   Subbox/smooth
#	CINT     Contour interval/min/max
#	LINE     Color/type/width/label/smth/fltr
#	FINT     Fill interval/min/max
#	FLINE    Fill colors/fill types
#	HILO     Color/symbol/rng/rad/cnt/intp
#	HLSYM    HILO txt size/posn/font/wdth/hw
#	CLRBAR   Color/ornt/anch/x;y/ln;wd/freq|text_info*
#	WIND     Wind symbol/siz/wdth/typ/hdsz
#	REFVEC   Mag;x;y;txtsiz/font/wdth/HW;labl
#	TITLE    Title color/line/title
#	TEXT     Size/fnt/wdth/brdr/N-rot/just/hw flg*
#	CLEAR    Clear screen flag
#	GAREA    Graphics area
#	PROJ     Map projection/angles/margins|drop flag
#	MAP      Map color/dash/width/filter flag
#	LATLON   Line color/dash/width/freq/inc/label/format
#	DEVICE   Device|name|x size;y size|color type
#	STNPLT   Txtc/txt attr|marker attr|stnfil#col
#	SATFIL   Satellite image filename(s)
#	RADFIL   Radar image filename(s)
#	LUTFIL   Enhancement lookup table filename
#	STREAM   lines/arrows/stop/slow/scale
#	POSN     Position
#	COLORS   Color list
#	MARKER   Marker color/type/size/width/hw
#	GRDLBL   Grid point label color
#	FILTER   Filter data factor
#
# Modifications: 3/20/2007 - ARK
#		  - Changed LATLON from 0 to a dashed line so it appears over the storm
#		  - Changed the TYPE from c/f to f so no contours are plotted only fill colors	
#-----------------------------------------------------------------------------
sub gdplot3{

	#--- Define local variables
	
	#vvvvvvvvvvvvvvvvvvvvvvvvvv
	#-- ARK - 8/16/2007 - Changed parameter list to include the $fill parameter which can be either 0 true or 1 false
        #my ($rPlot, $fhr, $function, $functionChar, $giffile, $rProbType, $titleFile, $labels, $GEMPAK, $LOG) = @_;
	my ($rPlot, $fhr, $function, $functionChar, $giffile, $rProbType, $titleFile, $labels, $fill, $GEMPAK, $LOG) = @_;
	#^^^^^^^^^^^^^^^^^^^^^^^^^^

print ($LOG "         Plotting wind probability contoured and filled data.\n"); 	
print $GEMPAK <<END_OF_TEXT;
	
gdplot3 <<EOF;
MAP     = 32//1 
LATLON  = 32/10/1/1;1/5;5/$rPlot->{latLonTextLoc}/2 
PROJ    = $rPlot->{proj}
GLEVEL  = 0
GVCORD  = none
GAREA   = $rPlot->{area}
GDFILE  = $rPlot->{gridFile}
GDATTIM = F$fhr
PANEL   = 0
SKIP    = 0
SCALE   = 0
GDPFUN  = $function
END_OF_TEXT

if ($fill == 1) {
print $GEMPAK "TYPE    = f\n";
}
else {
print $GEMPAK "TYPE = \n";
}

print $GEMPAK <<END_OF_TEXT;
CONTUR  = 1
CINT    = $functionChar->{cint}
LINE    = 32/1/1/0/3
FINT    = $functionChar->{fint}
FLINE   = $functionChar->{fline}
HILO    =
HLSYM   =
END_OF_TEXT

if ($gempakClrBar == 1) {

print $GEMPAK "CLRBAR  = $functionChar->{clrbar}\n";

}
else {

print $GEMPAK "CLRBAR  = 0 \n";

}

print $GEMPAK <<END_OF_TEXT;
WIND    =
REFVEC  =
TITLE   =
TEXT    = 1.0/21/2///hw
CLEAR   = no
DEVICE  = $rPlot->{device} | $giffile
END_OF_TEXT

# Copy from $GEMTBL these 2 tables for labeling on the graphic of states/countries.

system ("cp $ENV{GEMTBL}/stns/tpc_countries.tbl .");
system ("cp $ENV{GEMTBL}/stns/tpc_states.tbl .");

if ($labels eq "country") {
        print $GEMPAK "STNPLT  = 32/0.8/22////l||tpc_countries.tbl#3 \n";
	#print $GEMPAK "STNPLT  = 32/0.8/22////l|| $ENV{GEMTBL}/stns/tpc_countries.tbl#3 \n";
	#print $GEMPAK "STNPLT  = 32/0.8/22////l||/home/devel/data/tables/tpc_countries.tbl#3 \n";
	#print $GEMPAK "STNPLT  = 32/0.8/22////l||$ENV{HOME}/data/tables/tpc_countries.tbl#3 \n";
}
elsif ($labels eq "state") {
        print $GEMPAK "STNPLT  = 32/0.8/22////l||tpc_states.tbl#1 \n";
	#print $GEMPAK "STNPLT  = 32/0.8/22////l|| $ENV{GEMTBL}/stns/tpc_states.tbl#1 \n";
	#print $GEMPAK "STNPLT  = 32/0.8/22////l||/home/devel/data/tables/tpc_states.tbl#1 \n";
	#print $GEMPAK "STNPLT  = 32/0.8/22////l||$ENV{HOME}/data/tables/tpc_states.tbl#1 \n";
}
else {
	print $GEMPAK "STNPLT = \n";
}

print $GEMPAK <<END_OF_TEXT;
SATFIL  =
RADFIL  = 
STREAM  =
POSN    = 4
COLORS  = 2
MARKER  = 2
GRDLBL  = 5
LUTFIL  = none
FILTER  = $rPlot->{filter}
BOXLIN   =  
END_OF_TEXT

if ($gempakText == 1) {

print $GEMPAK <<END_OF_TEXT2; 
REGION   = VIEW
TXTCOL   = $rProbType->{color}
TXTYPE   = $rProbType->{text}
TXTFIL   = $titleFile
TXTLOC   = $rProbType->{location}
END_OF_TEXT2

}
else {

print $GEMPAK <<END_OF_TEXT2; 
REGION   = 
TXTCOL   = 
TXTYPE   = 
TXTFIL   = 
TXTLOC   = 
END_OF_TEXT2

}

print $GEMPAK <<END_OF_TEXT;
COLUMN   = 1
SHAPE    =  
INFO     =  
LOCI     =  
ANOTLN   =  
ANOTYP   =

run

EOF
END_OF_TEXT

}

#--------------------------------------------------------------------------------
#GDPLOT2 - 
#
#	INPUT PARAMETERS
#	GDFILE   Grid file
#	GDATTIM  Grid date/time
#	GLEVEL   Grid level
#	GVCORD   Grid vertical coordinate
#	PANEL    Panel loc/color/dash/width/regn
#	SKIP     Skip_cntr/skip_plt_x;skip_plt_y
#	SCALE    Scalar scale / vector scale
#	GDPFUN   Scalar grid or vector grid function
#	TYPE     GDPLOT2 function processing type: C/L/I/F/X/A/B/S/G/P
#	CONTUR   Subbox/smooth
#	CINT     Contour interval/min/max
#	LINE     Color/type/width/label/smth/fltr
#	FINT     Fill interval/min/max
#	FLINE    Fill colors/fill types
#	HILO     Color/symbol/rng/rad/cnt/intp
#	HLSYM    HILO txt size/posn/font/wdth/hw
#	CLRBAR   Color/ornt/anch/x;y/ln;wd/freq|text_info*
#	WIND     Wind symbol/siz/wdth/typ/hdsz
#	REFVEC   Mag;x;y;txtsiz/font/wdth/HW;labl
#	TITLE    Title color/line/title
#	TEXT     Size/fnt/wdth/brdr/N-rot/just/hw flg*
#	CLEAR    Clear screen flag
#	GAREA    Graphics area
#	PROJ     Map projection/angles/margins|drop flag
#	MAP      Map color/dash/width/filter flag
#	LATLON   Line color/dash/width/freq/inc/label/format
#	DEVICE   Device|name|x size;y size|color type
#	STNPLT   Txtc/txt attr|marker attr|stnfil#col
#	SATFIL   Satellite image filename(s)
#	RADFIL   Radar image filename(s)
#	LUTFIL   Enhancement lookup table filename
#	STREAM   lines/arrows/stop/slow/scale
#	POSN     Position
#	COLORS   Color list
#	MARKER   Marker color/type/size/width/hw
#	GRDLBL   Grid point label color
#	FILTER   Filter data factor
#	
#--------------------------------------------------------------------------------------
sub gdplot2{
#--- Define local variables

        my ($rPlot, $fhr, $function, $functionChar, $giffile, $GEMPAK, $LOG) = @_;

print ($LOG "         Plotting wind probability contoured and filled data.\n"); 		
print $GEMPAK <<END_OF_TEXT;
	
gdplot2 <<EOF;
MAP     = 32//1 
LATLON  = 0 
PROJ    = $rPlot->{proj}
GLEVEL  = 0
GVCORD  = none
GAREA   = $rPlot->{area}
GDFILE  = $rPlot->{gridFile}
GDATTIM = F$fhr
PANEL   = 0
SKIP    = 0
SCALE   = 0
GDPFUN  = $function
TYPE    = c/f 
CONTUR  = 1
CINT    = $functionChar->{cint}
LINE    = 32/1/1/0/3
FINT    = $functionChar->{fint}
FLINE   = $functionChar->{fline}
HILO    =
HLSYM   =
CLRBAR  = $functionChar->{clrbar}
WIND    =
REFVEC  =
TITLE   =
TEXT    = 1.0/21/2/hw
CLEAR   = no
DEVICE  = $rPlot->{device} | $giffile
STNPLT  =  
SATFIL  =
RADFIL  = 
STREAM  =
POSN    = 4
COLORS  = 2
MARKER  = 2
GRDLBL  = 5
LUTFIL  = none
FILTER  = $rPlot->{filter}

run

EOF

END_OF_TEXT

}

#--------------------------------------------------------------------------------------------------
#GPTEXT - add text and logos to plot
#
#	INPUT PARAMETERS
#	PANEL     Panel loc/color/dash/width/regn   
# 	*COLORS    Color list   (6|32)                      
# 	*TEXT      Size/fnt/wdth/brdr/N-rot/just/hw (  1.2/11/1/111/1/c/sw | 1.3/23/1/221/1/c/hw )
# 	CLEAR     Clear screen flag                 
# 	DEVICE    Device|name|x size;y size|color   
# 	*TXTFIL    Text filename or LOGO|size|mode
# 	*TXTLOC    Text location ($noaaloc , $nwsloc , $titleloc , $title2loc )   
# 	COLUMN    Number of columns                 
#	
#----------------------------------------------------------------------------------------------------
sub gptext{

	#--- Define local variables

        my ($rPlot, $rProbType, $file, $giffile, $dir, $GEMPAK, $LOG) = @_;

print ($LOG "           Putting text on the graphic.\n");	
print $GEMPAK <<END_OF_TEXT;
	
gptext <<EOF;
PANEL    = 0
COLORS   = $rProbType->{color}
TEXT     = $rProbType->{text}
CLEAR    = no
DEVICE   = $rPlot->{device} | $giffile
TXTFIL   = $dir$file
TXTLOC   = $rProbType->{location}
COLUMN   = 1

run

EOF

END_OF_TEXT

}

#-----------------------------------------------------------------------------------------------------
# ATEST - gemplot testing program which can be used to place markers 
#         and text on a plot.
#	
#-----------------------------------------------------------------------------------------------------
sub atest {

	#--- Define local variables

        my ($rPlot, $GEMPAK, $LOG) = @_;
	
print ($LOG "         Placing lat/lon storm center marker on graphic at $rPlot->{stormLocation} . \n");	
print $GEMPAK <<END_OF_TEXT;

atest <<EOF;
ginitp
0
gscolr
31
gsmrkr
20 0 1 3
gmark
m
1
$rPlot->{stormLocation}
gscolr
32
gsmrkr
6 0 1 2
gmark
m
1
$rPlot->{stormLocation}
geplot
exit
EOF

END_OF_TEXT

}


1;
